﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codebuild/CodeBuildRequest.h>
#include <aws/codebuild/CodeBuild_EXPORTS.h>
#include <aws/codebuild/model/SortOrderType.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodeBuild {
namespace Model {

/**
 */
class ListBuildsRequest : public CodeBuildRequest {
 public:
  AWS_CODEBUILD_API ListBuildsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListBuilds"; }

  AWS_CODEBUILD_API Aws::String SerializePayload() const override;

  AWS_CODEBUILD_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The order to list build IDs. Valid values include:</p> <ul> <li> <p>
   * <code>ASCENDING</code>: List the build IDs in ascending order by build ID.</p>
   * </li> <li> <p> <code>DESCENDING</code>: List the build IDs in descending order
   * by build ID.</p> </li> </ul>
   */
  inline SortOrderType GetSortOrder() const { return m_sortOrder; }
  inline bool SortOrderHasBeenSet() const { return m_sortOrderHasBeenSet; }
  inline void SetSortOrder(SortOrderType value) {
    m_sortOrderHasBeenSet = true;
    m_sortOrder = value;
  }
  inline ListBuildsRequest& WithSortOrder(SortOrderType value) {
    SetSortOrder(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>During a previous call, if there are more than 100 items in the list, only
   * the first 100 items are returned, along with a unique string called a
   * <i>nextToken</i>. To get the next batch of items in the list, call this
   * operation again, adding the next token to the call. To get all of the items in
   * the list, keep calling this operation with each subsequent next token that is
   * returned, until no more next tokens are returned.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListBuildsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  SortOrderType m_sortOrder{SortOrderType::NOT_SET};

  Aws::String m_nextToken;
  bool m_sortOrderHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeBuild
}  // namespace Aws
