﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/customer-profiles/CustomerProfiles_EXPORTS.h>
#include <aws/customer-profiles/model/Scope.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Json {
class JsonValue;
}  // namespace Json
}  // namespace Utils
namespace CustomerProfiles {
namespace Model {
class GetIntegrationResult {
 public:
  AWS_CUSTOMERPROFILES_API GetIntegrationResult() = default;
  AWS_CUSTOMERPROFILES_API GetIntegrationResult(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);
  AWS_CUSTOMERPROFILES_API GetIntegrationResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);

  ///@{
  /**
   * <p>The unique name of the domain.</p>
   */
  inline const Aws::String& GetDomainName() const { return m_domainName; }
  template <typename DomainNameT = Aws::String>
  void SetDomainName(DomainNameT&& value) {
    m_domainNameHasBeenSet = true;
    m_domainName = std::forward<DomainNameT>(value);
  }
  template <typename DomainNameT = Aws::String>
  GetIntegrationResult& WithDomainName(DomainNameT&& value) {
    SetDomainName(std::forward<DomainNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The URI of the S3 bucket or any other type of data source.</p>
   */
  inline const Aws::String& GetUri() const { return m_uri; }
  template <typename UriT = Aws::String>
  void SetUri(UriT&& value) {
    m_uriHasBeenSet = true;
    m_uri = std::forward<UriT>(value);
  }
  template <typename UriT = Aws::String>
  GetIntegrationResult& WithUri(UriT&& value) {
    SetUri(std::forward<UriT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the profile object type.</p>
   */
  inline const Aws::String& GetObjectTypeName() const { return m_objectTypeName; }
  template <typename ObjectTypeNameT = Aws::String>
  void SetObjectTypeName(ObjectTypeNameT&& value) {
    m_objectTypeNameHasBeenSet = true;
    m_objectTypeName = std::forward<ObjectTypeNameT>(value);
  }
  template <typename ObjectTypeNameT = Aws::String>
  GetIntegrationResult& WithObjectTypeName(ObjectTypeNameT&& value) {
    SetObjectTypeName(std::forward<ObjectTypeNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp of when the domain was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedAt() const { return m_createdAt; }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  void SetCreatedAt(CreatedAtT&& value) {
    m_createdAtHasBeenSet = true;
    m_createdAt = std::forward<CreatedAtT>(value);
  }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  GetIntegrationResult& WithCreatedAt(CreatedAtT&& value) {
    SetCreatedAt(std::forward<CreatedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp of when the domain was most recently edited.</p>
   */
  inline const Aws::Utils::DateTime& GetLastUpdatedAt() const { return m_lastUpdatedAt; }
  template <typename LastUpdatedAtT = Aws::Utils::DateTime>
  void SetLastUpdatedAt(LastUpdatedAtT&& value) {
    m_lastUpdatedAtHasBeenSet = true;
    m_lastUpdatedAt = std::forward<LastUpdatedAtT>(value);
  }
  template <typename LastUpdatedAtT = Aws::Utils::DateTime>
  GetIntegrationResult& WithLastUpdatedAt(LastUpdatedAtT&& value) {
    SetLastUpdatedAt(std::forward<LastUpdatedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags used to organize, track, or control access for this resource.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  GetIntegrationResult& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  GetIntegrationResult& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A map in which each key is an event type from an external application such as
   * Segment or Shopify, and each value is an <code>ObjectTypeName</code> (template)
   * used to ingest the event. It supports the following event types:
   * <code>SegmentIdentify</code>, <code>ShopifyCreateCustomers</code>,
   * <code>ShopifyUpdateCustomers</code>, <code>ShopifyCreateDraftOrders</code>,
   * <code>ShopifyUpdateDraftOrders</code>, <code>ShopifyCreateOrders</code>, and
   * <code>ShopifyUpdatedOrders</code>.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetObjectTypeNames() const { return m_objectTypeNames; }
  template <typename ObjectTypeNamesT = Aws::Map<Aws::String, Aws::String>>
  void SetObjectTypeNames(ObjectTypeNamesT&& value) {
    m_objectTypeNamesHasBeenSet = true;
    m_objectTypeNames = std::forward<ObjectTypeNamesT>(value);
  }
  template <typename ObjectTypeNamesT = Aws::Map<Aws::String, Aws::String>>
  GetIntegrationResult& WithObjectTypeNames(ObjectTypeNamesT&& value) {
    SetObjectTypeNames(std::forward<ObjectTypeNamesT>(value));
    return *this;
  }
  template <typename ObjectTypeNamesKeyT = Aws::String, typename ObjectTypeNamesValueT = Aws::String>
  GetIntegrationResult& AddObjectTypeNames(ObjectTypeNamesKeyT&& key, ObjectTypeNamesValueT&& value) {
    m_objectTypeNamesHasBeenSet = true;
    m_objectTypeNames.emplace(std::forward<ObjectTypeNamesKeyT>(key), std::forward<ObjectTypeNamesValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unique identifier for the workflow.</p>
   */
  inline const Aws::String& GetWorkflowId() const { return m_workflowId; }
  template <typename WorkflowIdT = Aws::String>
  void SetWorkflowId(WorkflowIdT&& value) {
    m_workflowIdHasBeenSet = true;
    m_workflowId = std::forward<WorkflowIdT>(value);
  }
  template <typename WorkflowIdT = Aws::String>
  GetIntegrationResult& WithWorkflowId(WorkflowIdT&& value) {
    SetWorkflowId(std::forward<WorkflowIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Boolean that shows if the Flow that's associated with the Integration is
   * created in Amazon Appflow, or with ObjectTypeName equals _unstructured via
   * API/CLI in flowDefinition.</p>
   */
  inline bool GetIsUnstructured() const { return m_isUnstructured; }
  inline void SetIsUnstructured(bool value) {
    m_isUnstructuredHasBeenSet = true;
    m_isUnstructured = value;
  }
  inline GetIntegrationResult& WithIsUnstructured(bool value) {
    SetIsUnstructured(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the IAM role. The Integration uses this
   * role to make Customer Profiles requests on your behalf.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  GetIntegrationResult& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of unique names for active event triggers associated with the
   * integration. This list would be empty if no Event Trigger is associated with the
   * integration.</p>
   */
  inline const Aws::Vector<Aws::String>& GetEventTriggerNames() const { return m_eventTriggerNames; }
  template <typename EventTriggerNamesT = Aws::Vector<Aws::String>>
  void SetEventTriggerNames(EventTriggerNamesT&& value) {
    m_eventTriggerNamesHasBeenSet = true;
    m_eventTriggerNames = std::forward<EventTriggerNamesT>(value);
  }
  template <typename EventTriggerNamesT = Aws::Vector<Aws::String>>
  GetIntegrationResult& WithEventTriggerNames(EventTriggerNamesT&& value) {
    SetEventTriggerNames(std::forward<EventTriggerNamesT>(value));
    return *this;
  }
  template <typename EventTriggerNamesT = Aws::String>
  GetIntegrationResult& AddEventTriggerNames(EventTriggerNamesT&& value) {
    m_eventTriggerNamesHasBeenSet = true;
    m_eventTriggerNames.emplace_back(std::forward<EventTriggerNamesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether the integration applies to profile level data (associated
   * with profiles) or domain level data (not associated with any specific profile).
   * The default value is PROFILE.</p>
   */
  inline Scope GetScope() const { return m_scope; }
  inline void SetScope(Scope value) {
    m_scopeHasBeenSet = true;
    m_scope = value;
  }
  inline GetIntegrationResult& WithScope(Scope value) {
    SetScope(value);
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  GetIntegrationResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_domainName;

  Aws::String m_uri;

  Aws::String m_objectTypeName;

  Aws::Utils::DateTime m_createdAt{};

  Aws::Utils::DateTime m_lastUpdatedAt{};

  Aws::Map<Aws::String, Aws::String> m_tags;

  Aws::Map<Aws::String, Aws::String> m_objectTypeNames;

  Aws::String m_workflowId;

  bool m_isUnstructured{false};

  Aws::String m_roleArn;

  Aws::Vector<Aws::String> m_eventTriggerNames;

  Scope m_scope{Scope::NOT_SET};

  Aws::String m_requestId;
  bool m_domainNameHasBeenSet = false;
  bool m_uriHasBeenSet = false;
  bool m_objectTypeNameHasBeenSet = false;
  bool m_createdAtHasBeenSet = false;
  bool m_lastUpdatedAtHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_objectTypeNamesHasBeenSet = false;
  bool m_workflowIdHasBeenSet = false;
  bool m_isUnstructuredHasBeenSet = false;
  bool m_roleArnHasBeenSet = false;
  bool m_eventTriggerNamesHasBeenSet = false;
  bool m_scopeHasBeenSet = false;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace CustomerProfiles
}  // namespace Aws
