﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/resiliencehub/ResilienceHub_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace ResilienceHub {
namespace Model {

/**
 * <p>Indicates a specific risk identified in the Resilience Hub assessment and the
 * corresponding recommendation provided to address that risk.</p>  <p>The
 * assessment summary generated by large language models (LLMs) on Amazon Bedrock
 * are only suggestions. The current level of generative AI technology is not
 * perfect and LLMs are not infallible. Bias and incorrect answers, although rare,
 * should be expected. Review each recommendation in the assessment summary before
 * you use the output from an LLM. </p>   <p>This property is
 * available only in the US East (N. Virginia) Region.</p> <p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/resiliencehub-2020-04-30/AssessmentRiskRecommendation">AWS
 * API Reference</a></p>
 */
class AssessmentRiskRecommendation {
 public:
  AWS_RESILIENCEHUB_API AssessmentRiskRecommendation() = default;
  AWS_RESILIENCEHUB_API AssessmentRiskRecommendation(Aws::Utils::Json::JsonView jsonValue);
  AWS_RESILIENCEHUB_API AssessmentRiskRecommendation& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_RESILIENCEHUB_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>Indicates the Application Components (AppComponents) that were assessed as
   * part of the assessment and are associated with the identified risk and
   * recommendation.</p>  <p>This property is available only in the US East (N.
   * Virginia) Region.</p>
   */
  inline const Aws::Vector<Aws::String>& GetAppComponents() const { return m_appComponents; }
  inline bool AppComponentsHasBeenSet() const { return m_appComponentsHasBeenSet; }
  template <typename AppComponentsT = Aws::Vector<Aws::String>>
  void SetAppComponents(AppComponentsT&& value) {
    m_appComponentsHasBeenSet = true;
    m_appComponents = std::forward<AppComponentsT>(value);
  }
  template <typename AppComponentsT = Aws::Vector<Aws::String>>
  AssessmentRiskRecommendation& WithAppComponents(AppComponentsT&& value) {
    SetAppComponents(std::forward<AppComponentsT>(value));
    return *this;
  }
  template <typename AppComponentsT = Aws::String>
  AssessmentRiskRecommendation& AddAppComponents(AppComponentsT&& value) {
    m_appComponentsHasBeenSet = true;
    m_appComponents.emplace_back(std::forward<AppComponentsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates the recommendation provided by the Resilience Hub to address the
   * identified risks in the application.</p>  <p>This property is available
   * only in the US East (N. Virginia) Region.</p>
   */
  inline const Aws::String& GetRecommendation() const { return m_recommendation; }
  inline bool RecommendationHasBeenSet() const { return m_recommendationHasBeenSet; }
  template <typename RecommendationT = Aws::String>
  void SetRecommendation(RecommendationT&& value) {
    m_recommendationHasBeenSet = true;
    m_recommendation = std::forward<RecommendationT>(value);
  }
  template <typename RecommendationT = Aws::String>
  AssessmentRiskRecommendation& WithRecommendation(RecommendationT&& value) {
    SetRecommendation(std::forward<RecommendationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates the description of the potential risk identified in the application
   * as part of the Resilience Hub assessment.</p>  <p>This property is
   * available only in the US East (N. Virginia) Region.</p>
   */
  inline const Aws::String& GetRisk() const { return m_risk; }
  inline bool RiskHasBeenSet() const { return m_riskHasBeenSet; }
  template <typename RiskT = Aws::String>
  void SetRisk(RiskT&& value) {
    m_riskHasBeenSet = true;
    m_risk = std::forward<RiskT>(value);
  }
  template <typename RiskT = Aws::String>
  AssessmentRiskRecommendation& WithRisk(RiskT&& value) {
    SetRisk(std::forward<RiskT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_appComponents;

  Aws::String m_recommendation;

  Aws::String m_risk;
  bool m_appComponentsHasBeenSet = false;
  bool m_recommendationHasBeenSet = false;
  bool m_riskHasBeenSet = false;
};

}  // namespace Model
}  // namespace ResilienceHub
}  // namespace Aws
