﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/serverlessrepo/ServerlessApplicationRepositoryRequest.h>
#include <aws/serverlessrepo/ServerlessApplicationRepository_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ServerlessApplicationRepository {
namespace Model {

/**
 */
class CreateApplicationVersionRequest : public ServerlessApplicationRepositoryRequest {
 public:
  AWS_SERVERLESSAPPLICATIONREPOSITORY_API CreateApplicationVersionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateApplicationVersion"; }

  AWS_SERVERLESSAPPLICATIONREPOSITORY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the application.</p>
   */
  inline const Aws::String& GetApplicationId() const { return m_applicationId; }
  inline bool ApplicationIdHasBeenSet() const { return m_applicationIdHasBeenSet; }
  template <typename ApplicationIdT = Aws::String>
  void SetApplicationId(ApplicationIdT&& value) {
    m_applicationIdHasBeenSet = true;
    m_applicationId = std::forward<ApplicationIdT>(value);
  }
  template <typename ApplicationIdT = Aws::String>
  CreateApplicationVersionRequest& WithApplicationId(ApplicationIdT&& value) {
    SetApplicationId(std::forward<ApplicationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The semantic version of the new version.</p>
   */
  inline const Aws::String& GetSemanticVersion() const { return m_semanticVersion; }
  inline bool SemanticVersionHasBeenSet() const { return m_semanticVersionHasBeenSet; }
  template <typename SemanticVersionT = Aws::String>
  void SetSemanticVersion(SemanticVersionT&& value) {
    m_semanticVersionHasBeenSet = true;
    m_semanticVersion = std::forward<SemanticVersionT>(value);
  }
  template <typename SemanticVersionT = Aws::String>
  CreateApplicationVersionRequest& WithSemanticVersion(SemanticVersionT&& value) {
    SetSemanticVersion(std::forward<SemanticVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A link to the S3 object that contains the ZIP archive of the source code for
   * this version of your application.</p><p>Maximum size 50 MB</p>
   */
  inline const Aws::String& GetSourceCodeArchiveUrl() const { return m_sourceCodeArchiveUrl; }
  inline bool SourceCodeArchiveUrlHasBeenSet() const { return m_sourceCodeArchiveUrlHasBeenSet; }
  template <typename SourceCodeArchiveUrlT = Aws::String>
  void SetSourceCodeArchiveUrl(SourceCodeArchiveUrlT&& value) {
    m_sourceCodeArchiveUrlHasBeenSet = true;
    m_sourceCodeArchiveUrl = std::forward<SourceCodeArchiveUrlT>(value);
  }
  template <typename SourceCodeArchiveUrlT = Aws::String>
  CreateApplicationVersionRequest& WithSourceCodeArchiveUrl(SourceCodeArchiveUrlT&& value) {
    SetSourceCodeArchiveUrl(std::forward<SourceCodeArchiveUrlT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A link to a public repository for the source code of your application, for
   * example the URL of a specific GitHub commit.</p>
   */
  inline const Aws::String& GetSourceCodeUrl() const { return m_sourceCodeUrl; }
  inline bool SourceCodeUrlHasBeenSet() const { return m_sourceCodeUrlHasBeenSet; }
  template <typename SourceCodeUrlT = Aws::String>
  void SetSourceCodeUrl(SourceCodeUrlT&& value) {
    m_sourceCodeUrlHasBeenSet = true;
    m_sourceCodeUrl = std::forward<SourceCodeUrlT>(value);
  }
  template <typename SourceCodeUrlT = Aws::String>
  CreateApplicationVersionRequest& WithSourceCodeUrl(SourceCodeUrlT&& value) {
    SetSourceCodeUrl(std::forward<SourceCodeUrlT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The raw packaged AWS SAM template of your application.</p>
   */
  inline const Aws::String& GetTemplateBody() const { return m_templateBody; }
  inline bool TemplateBodyHasBeenSet() const { return m_templateBodyHasBeenSet; }
  template <typename TemplateBodyT = Aws::String>
  void SetTemplateBody(TemplateBodyT&& value) {
    m_templateBodyHasBeenSet = true;
    m_templateBody = std::forward<TemplateBodyT>(value);
  }
  template <typename TemplateBodyT = Aws::String>
  CreateApplicationVersionRequest& WithTemplateBody(TemplateBodyT&& value) {
    SetTemplateBody(std::forward<TemplateBodyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A link to the packaged AWS SAM template of your application.</p>
   */
  inline const Aws::String& GetTemplateUrl() const { return m_templateUrl; }
  inline bool TemplateUrlHasBeenSet() const { return m_templateUrlHasBeenSet; }
  template <typename TemplateUrlT = Aws::String>
  void SetTemplateUrl(TemplateUrlT&& value) {
    m_templateUrlHasBeenSet = true;
    m_templateUrl = std::forward<TemplateUrlT>(value);
  }
  template <typename TemplateUrlT = Aws::String>
  CreateApplicationVersionRequest& WithTemplateUrl(TemplateUrlT&& value) {
    SetTemplateUrl(std::forward<TemplateUrlT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationId;

  Aws::String m_semanticVersion;

  Aws::String m_sourceCodeArchiveUrl;

  Aws::String m_sourceCodeUrl;

  Aws::String m_templateBody;

  Aws::String m_templateUrl;
  bool m_applicationIdHasBeenSet = false;
  bool m_semanticVersionHasBeenSet = false;
  bool m_sourceCodeArchiveUrlHasBeenSet = false;
  bool m_sourceCodeUrlHasBeenSet = false;
  bool m_templateBodyHasBeenSet = false;
  bool m_templateUrlHasBeenSet = false;
};

}  // namespace Model
}  // namespace ServerlessApplicationRepository
}  // namespace Aws
