# SPDX-FileCopyrightText: Copyright (c) 2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
# SPDX-License-Identifier: LicenseRef-NvidiaProprietary
#
# NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
# property and proprietary rights in and to this material, related
# documentation and any modifications thereto. Any use, reproduction,
# disclosure or distribution of this material and related documentation
# without an express license agreement from NVIDIA CORPORATION or
# its affiliates is strictly prohibited.

import importlib.metadata as metadata
import os
import re
import subprocess
import sys
from pathlib import Path
from typing import TYPE_CHECKING

if TYPE_CHECKING:
    # Use constants from  installed package
    from nsys_recipe import nsys_constants
else:
    # Normally the script is just invoked with a absolute path given,
    # so we can't use the actual package.
    import nsys_constants


def run_python_command(cmd: str) -> None:
    args = [sys.executable, "-m"] + cmd.split(" ")
    try:
        subprocess.run(args, check=True, text=True)
    except subprocess.CalledProcessError as e:
        if "mypy" not in cmd or "exit status 1" not in str(e):
            print(e)
        sys.exit(1)


def extract_package_name(line: str) -> str:
    # Remove comments.
    line = line.split("#", 1)[0].strip()

    # Match the package name at the start of the line
    match = re.match(r"^[A-Za-z0-9_.-]+", line)
    return match.group(0) if match else ""


def check_dependencies() -> None:
    with open(nsys_constants.NSYS_RECIPE_REQ_LINTING_PATH) as f:
        for line in f:
            package_name = extract_package_name(line)
            if not package_name:
                continue

            try:
                metadata.version(package_name)
            except metadata.PackageNotFoundError:
                print(
                    f"Package `{package_name}` not found."
                    f" Please run `{nsys_constants.NSYS_RECIPE_INSTALL_PATH}` with the `--linting` option"
                    " to install the necessary dependencies."
                )
                sys.exit(1)


def format_and_validate_code() -> None:
    recipe_pkg_dir = Path(__file__).parent
    os.chdir(recipe_pkg_dir)

    tools = ["isort .", "black .", "mypy ."]

    for tool in tools:
        run_python_command(tool)


def main():
    check_dependencies()
    format_and_validate_code()


if __name__ == "__main__":
    main()
