package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link ImageHistory}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableImageHistory.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableImageHistory implements ImageHistory {
  private final String id;
  private final Long created;
  private final String createdBy;
  private final @Nullable List<String> tags;
  private final Long size;
  private final @Nullable String comment;

  private ImmutableImageHistory(
      String id,
      Long created,
      String createdBy,
      @Nullable List<String> tags,
      Long size,
      @Nullable String comment) {
    this.id = id;
    this.created = created;
    this.createdBy = createdBy;
    this.tags = tags;
    this.size = size;
    this.comment = comment;
  }

  /**
   * @return The value of the {@code id} attribute
   */
  @JsonProperty("Id")
  @Override
  public String id() {
    return id;
  }

  /**
   * @return The value of the {@code created} attribute
   */
  @JsonProperty("Created")
  @Override
  public Long created() {
    return created;
  }

  /**
   * @return The value of the {@code createdBy} attribute
   */
  @JsonProperty("CreatedBy")
  @Override
  public String createdBy() {
    return createdBy;
  }

  /**
   * @return The value of the {@code tags} attribute
   */
  @JsonProperty("Tags")
  @Override
  public @Nullable List<String> tags() {
    return tags;
  }

  /**
   * @return The value of the {@code size} attribute
   */
  @JsonProperty("Size")
  @Override
  public Long size() {
    return size;
  }

  /**
   * @return The value of the {@code comment} attribute
   */
  @JsonProperty("Comment")
  @Override
  public @Nullable String comment() {
    return comment;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ImageHistory#id() id} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for id
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableImageHistory withId(String value) {
    String newValue = Objects.requireNonNull(value, "id");
    if (this.id.equals(newValue)) return this;
    return new ImmutableImageHistory(newValue, this.created, this.createdBy, this.tags, this.size, this.comment);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ImageHistory#created() created} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for created
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableImageHistory withCreated(Long value) {
    Long newValue = Objects.requireNonNull(value, "created");
    if (this.created.equals(newValue)) return this;
    return new ImmutableImageHistory(this.id, newValue, this.createdBy, this.tags, this.size, this.comment);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ImageHistory#createdBy() createdBy} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for createdBy
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableImageHistory withCreatedBy(String value) {
    String newValue = Objects.requireNonNull(value, "createdBy");
    if (this.createdBy.equals(newValue)) return this;
    return new ImmutableImageHistory(this.id, this.created, newValue, this.tags, this.size, this.comment);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ImageHistory#tags() tags}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableImageHistory withTags(@Nullable String... elements) {
    if (elements == null) {
      if (this.tags == null) return this;
      return new ImmutableImageHistory(this.id, this.created, this.createdBy, null, this.size, this.comment);
    }
    List<String> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableImageHistory(this.id, this.created, this.createdBy, newValue, this.size, this.comment);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ImageHistory#tags() tags}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of tags elements to set
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableImageHistory withTags(@Nullable Iterable<String> elements) {
    if (this.tags == elements) return this;
    @Nullable List<String> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableImageHistory(this.id, this.created, this.createdBy, newValue, this.size, this.comment);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ImageHistory#size() size} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for size
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableImageHistory withSize(Long value) {
    Long newValue = Objects.requireNonNull(value, "size");
    if (this.size.equals(newValue)) return this;
    return new ImmutableImageHistory(this.id, this.created, this.createdBy, this.tags, newValue, this.comment);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ImageHistory#comment() comment} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for comment (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableImageHistory withComment(@Nullable String value) {
    if (Objects.equals(this.comment, value)) return this;
    return new ImmutableImageHistory(this.id, this.created, this.createdBy, this.tags, this.size, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableImageHistory} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableImageHistory
        && equalsByValue((ImmutableImageHistory) another);
  }

  private boolean equalsByValue(ImmutableImageHistory another) {
    return id.equals(another.id)
        && created.equals(another.created)
        && createdBy.equals(another.createdBy)
        && Objects.equals(tags, another.tags)
        && size.equals(another.size)
        && Objects.equals(comment, another.comment);
  }

  /**
   * Computes a hash code from attributes: {@code id}, {@code created}, {@code createdBy}, {@code tags}, {@code size}, {@code comment}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + id.hashCode();
    h += (h << 5) + created.hashCode();
    h += (h << 5) + createdBy.hashCode();
    h += (h << 5) + Objects.hashCode(tags);
    h += (h << 5) + size.hashCode();
    h += (h << 5) + Objects.hashCode(comment);
    return h;
  }

  /**
   * Prints the immutable value {@code ImageHistory} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ImageHistory{"
        + "id=" + id
        + ", created=" + created
        + ", createdBy=" + createdBy
        + ", tags=" + tags
        + ", size=" + size
        + ", comment=" + comment
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link ImageHistory} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ImageHistory instance
   */
  public static ImmutableImageHistory copyOf(ImageHistory instance) {
    if (instance instanceof ImmutableImageHistory) {
      return (ImmutableImageHistory) instance;
    }
    return ImmutableImageHistory.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableImageHistory ImmutableImageHistory}.
   * <pre>
   * ImmutableImageHistory.builder()
   *    .id(String) // required {@link ImageHistory#id() id}
   *    .created(Long) // required {@link ImageHistory#created() created}
   *    .createdBy(String) // required {@link ImageHistory#createdBy() createdBy}
   *    .tags(List&lt;String&gt; | null) // nullable {@link ImageHistory#tags() tags}
   *    .size(Long) // required {@link ImageHistory#size() size}
   *    .comment(String | null) // nullable {@link ImageHistory#comment() comment}
   *    .build();
   * </pre>
   * @return A new ImmutableImageHistory builder
   */
  public static ImmutableImageHistory.Builder builder() {
    return new ImmutableImageHistory.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableImageHistory ImmutableImageHistory}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_ID = 0x1L;
    private static final long INIT_BIT_CREATED = 0x2L;
    private static final long INIT_BIT_CREATED_BY = 0x4L;
    private static final long INIT_BIT_SIZE = 0x8L;
    private long initBits = 0xfL;

    private String id;
    private Long created;
    private String createdBy;
    private List<String> tags = null;
    private Long size;
    private String comment;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ImageHistory} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ImageHistory instance) {
      Objects.requireNonNull(instance, "instance");
      this.id(instance.id());
      this.created(instance.created());
      this.createdBy(instance.createdBy());
      @Nullable List<String> tagsValue = instance.tags();
      if (tagsValue != null) {
        addAllTags(tagsValue);
      }
      this.size(instance.size());
      @Nullable String commentValue = instance.comment();
      if (commentValue != null) {
        comment(commentValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ImageHistory#id() id} attribute.
     * @param id The value for id 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Id")
    public final Builder id(String id) {
      this.id = Objects.requireNonNull(id, "id");
      initBits &= ~INIT_BIT_ID;
      return this;
    }

    /**
     * Initializes the value for the {@link ImageHistory#created() created} attribute.
     * @param created The value for created 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Created")
    public final Builder created(Long created) {
      this.created = Objects.requireNonNull(created, "created");
      initBits &= ~INIT_BIT_CREATED;
      return this;
    }

    /**
     * Initializes the value for the {@link ImageHistory#createdBy() createdBy} attribute.
     * @param createdBy The value for createdBy 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("CreatedBy")
    public final Builder createdBy(String createdBy) {
      this.createdBy = Objects.requireNonNull(createdBy, "createdBy");
      initBits &= ~INIT_BIT_CREATED_BY;
      return this;
    }

    /**
     * Adds one element to {@link ImageHistory#tags() tags} list.
     * @param element A tags element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder tag(String element) {
      if (this.tags == null) {
        this.tags = new ArrayList<String>();
      }
      this.tags.add(Objects.requireNonNull(element, "tags element"));
      return this;
    }

    /**
     * Adds elements to {@link ImageHistory#tags() tags} list.
     * @param elements An array of tags elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder tags(String... elements) {
      if (this.tags == null) {
        this.tags = new ArrayList<String>();
      }
      for (String element : elements) {
        this.tags.add(Objects.requireNonNull(element, "tags element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link ImageHistory#tags() tags} list.
     * @param elements An iterable of tags elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Tags")
    public final Builder tags(@Nullable Iterable<String> elements) {
      if (elements == null) {
        this.tags = null;
        return this;
      }
      this.tags = new ArrayList<String>();
      return addAllTags(elements);
    }

    /**
     * Adds elements to {@link ImageHistory#tags() tags} list.
     * @param elements An iterable of tags elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllTags(Iterable<String> elements) {
      Objects.requireNonNull(elements, "tags element");
      if (this.tags == null) {
        this.tags = new ArrayList<String>();
      }
      for (String element : elements) {
        this.tags.add(Objects.requireNonNull(element, "tags element"));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ImageHistory#size() size} attribute.
     * @param size The value for size 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Size")
    public final Builder size(Long size) {
      this.size = Objects.requireNonNull(size, "size");
      initBits &= ~INIT_BIT_SIZE;
      return this;
    }

    /**
     * Initializes the value for the {@link ImageHistory#comment() comment} attribute.
     * @param comment The value for comment (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Comment")
    public final Builder comment(@Nullable String comment) {
      this.comment = comment;
      return this;
    }

    /**
     * Builds a new {@link ImmutableImageHistory ImmutableImageHistory}.
     * @return An immutable instance of ImageHistory
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableImageHistory build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableImageHistory(
          id,
          created,
          createdBy,
          tags == null ? null : createUnmodifiableList(true, tags),
          size,
          comment);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_ID) != 0) attributes.add("id");
      if ((initBits & INIT_BIT_CREATED) != 0) attributes.add("created");
      if ((initBits & INIT_BIT_CREATED_BY) != 0) attributes.add("createdBy");
      if ((initBits & INIT_BIT_SIZE) != 0) attributes.add("size");
      return "Cannot build ImageHistory, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<? extends T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
