/*******************************************************************************
 * Copyright (c) 2005 - 2006 Joel Cheuoua & others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Joel Cheuoua - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.codegen.jet.editor.codeassist;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.net.MalformedURLException;
import java.util.Arrays;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.codegen.jet.JETException;
import org.eclipse.emf.codegen.jet.JETNature;
import org.eclipse.emf.codegen.jet.JETSkeleton;
import org.eclipse.emf.codegen.jet.editor.JETEditorPlugin;
import org.eclipse.emf.codegen.jet.editor.presentation.JETTextEditor;
import org.eclipse.emf.codegen.jet.editor.util.JETCompilerExt;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.corext.template.java.JavaContextType;
import org.eclipse.jdt.internal.ui.JavaPlugin;
import org.eclipse.jdt.internal.ui.text.java.JavaCompletionProposal;
import org.eclipse.jdt.internal.ui.text.java.LazyJavaCompletionProposal;
import org.eclipse.jdt.internal.ui.text.template.contentassist.TemplateEngine;
import org.eclipse.jdt.internal.ui.text.template.contentassist.TemplateProposal;
import org.eclipse.jdt.ui.text.java.CompletionProposalCollector;
import org.eclipse.jdt.ui.text.java.CompletionProposalComparator;
import org.eclipse.jdt.ui.text.java.IJavaCompletionProposal;
import org.eclipse.jface.text.ITextViewer;
import org.eclipse.jface.text.contentassist.ICompletionProposal;
import org.eclipse.jface.text.contentassist.IContentAssistProcessor;
import org.eclipse.jface.text.contentassist.IContextInformation;
import org.eclipse.jface.text.contentassist.IContextInformationValidator;
import org.eclipse.jface.text.templates.TemplateContextType;
import org.eclipse.swt.graphics.Point;
import org.eclipse.ui.IFileEditorInput;

/**
 * @author Joel Cheuoua
 * @version $Revision: 1.2 $
 */
public class JETTemplateCompletionProcessor implements IContentAssistProcessor {
  protected IFile jetFile;
  protected JETTextEditor jetEditor;
  protected IJavaProject javaProject;
  protected JETNature jetNature;
  private TemplateEngine fTemplateEngine;
  private CompletionProposalComparator fComparator;
  private char[] fProposalAutoActivationSet;
  
  /**
   * Constructor for JETTemplateCompletionProcessor.
   * @param editor JETTextEditor
   */
  public JETTemplateCompletionProcessor(JETTextEditor editor) {
    this.jetEditor = editor;
    setCompletionProposalAutoActivationCharacters(".".toCharArray());
    jetFile = ((IFileEditorInput) editor.getEditorInput()).getFile();
    javaProject = JavaCore.create(getProject());
    jetNature = JETNature.getRuntime(getProject());
    TemplateContextType contextType = JavaPlugin.getDefault().getTemplateContextRegistry().getContextType("java"); //$NON-NLS-1$
    if (contextType == null) {
      contextType = new JavaContextType();
      JavaPlugin.getDefault().getTemplateContextRegistry().addContextType(contextType);
    }
    if (contextType != null)
      fTemplateEngine = new TemplateEngine(contextType);
    fComparator= new CompletionProposalComparator();
  }

  /**
   * Sets this processor's set of characters triggering the activation of the
   * completion proposal computation.
   * 
   * @param activationSet the activation set
   */
  public void setCompletionProposalAutoActivationCharacters(char[] activationSet) {
    fProposalAutoActivationSet= activationSet;
  }
  
  /**
   * Method getProject.
   * @return IProject
   */
  private IProject getProject() {
    if (jetFile == null)
      return null;
    return jetFile.getProject();
  }

  /**
   * Method getJavaProject.
   * @return IJavaProject
   */
  protected IJavaProject getJavaProject() {
    if (javaProject == null)
      javaProject = JavaCore.create(getProject());
    return javaProject;
  }

  /**
   * Method getJETNature.
   * @return JETNature
   */
  protected JETNature getJETNature() {
    if (jetNature == null)
      jetNature = JETNature.getRuntime(getProject());
    return jetNature;
  }

  /**
   * @param compiler
   * @param javaSource String
   * @param viewer ITextViewer
   * @param offset
   * @return int
   */
  protected int findJavaMatchingOffset(JETCompilerExt compiler, String javaSource, ITextViewer viewer, int offset) {
    JETCompilerExt.Range jetRange = compiler.getJETRange(offset, jetFile.getFullPath().toString());
    JETCompilerExt.Range javaRange = compiler.getJavaRange(jetRange);

    int javaOffset = (javaRange == null) ? -1 : javaRange.start + (offset - jetRange.start);
    if (javaRange == null)
      return -1;
    return javaOffset;
  }

  /**
   * @see IContentAssistProcessor#computeCompletionProposals(ITextViewer, int)
   */
  public ICompletionProposal[] computeCompletionProposals(ITextViewer viewer, int offset) {
    ICompletionProposal[] results;
    try {
      if (getJavaProject() == null || getJETNature() == null)
        return results = new ICompletionProposal[0];
      IContainer parent = jetFile.getParent();
      while (parent != null && !(jetNature.getTemplateContainers().contains(parent))) {          
        parent = parent.getParent();
      }
      if (parent == null) {
        jetNature.getTemplateContainers().add(jetFile.getParent());
        jetNature.setTemplateContainers(jetNature.getTemplateContainers());
      }
      if (jetNature.getJavaSourceContainer() == null)
        computeJavaSourceContainer();      

      // Ensure the template java class is generated
      //jetEditor.getSite().getPage().saveEditor(jetEditor, false);
      String uri = jetFile.getLocation().toFile().getAbsoluteFile().toURL().toString();
      String jetSource = viewer.getDocument().get();
      ByteArrayInputStream is = new ByteArrayInputStream(jetSource.getBytes());
      JETCompilerExt jetCompiler = new JETCompilerExt(uri, is);
      jetCompiler.parse();
      ByteArrayOutputStream arrayOutputStream = new ByteArrayOutputStream();
      jetCompiler.generate(arrayOutputStream);

      JETSkeleton skeleton = jetCompiler.getSkeleton();
      IPackageFragmentRoot[] roots = getJavaProject().getPackageFragmentRoots();
      ICompilationUnit compilationUnit = findCompilationUnit(arrayOutputStream, skeleton, roots);

      int javaOffset = findJavaMatchingOffset(jetCompiler, arrayOutputStream.toString(), viewer, offset);
      if (javaOffset == -1)
        return null;

      CompletionProposalCollector collector = new CompletionProposalCollector(compilationUnit);
      
      Point selection = viewer.getSelectedRange();
      if (selection.y > 0)
        collector.setReplacementLength(selection.y);

      compilationUnit.codeComplete(javaOffset, collector);
      results= collector.getJavaCompletionProposals();

      if (fTemplateEngine != null)
        results = adjustResults(viewer, offset, results, compilationUnit, collector);      

      viewer.setSelectedRange(offset, 0);
      viewer.revealRange(offset, 10);
      setReplacementOffsets(offset, results);      
      return order(results);
    } catch (CoreException e) {
      JETEditorPlugin.getDefault().log(e);
    } catch (MalformedURLException e) {
      JETEditorPlugin.getDefault().log(e);
    } catch (JETException e) {
      JETEditorPlugin.getDefault().log(e);
    } 
    return null;
  }

  private void setReplacementOffsets(int offset, ICompletionProposal[] results) {
    for (int i = 0; i < results.length; i++) {
      if (results[i] instanceof JavaCompletionProposal) {
        JavaCompletionProposal proposal = (JavaCompletionProposal) results[i];
        proposal.setReplacementOffset(offset);
      }
      if (results[i] instanceof LazyJavaCompletionProposal) {
        LazyJavaCompletionProposal proposal = (LazyJavaCompletionProposal) results[i];
        proposal.setReplacementOffset(offset);
      }        
    }
  }

  private ICompletionProposal[] adjustResults(ITextViewer viewer, int offset, ICompletionProposal[] results, ICompilationUnit compilationUnit, CompletionProposalCollector collector) {
    fTemplateEngine.reset();
    fTemplateEngine.complete(viewer, offset, compilationUnit);

    TemplateProposal[] templateResults = fTemplateEngine.getResults();

    // update relevance of template proposals that match with a keyword
    IJavaCompletionProposal[] keyWordResults = collector.getKeywordCompletionProposals();
    for (int i = 0; i < keyWordResults.length; i++) {
      String keyword = keyWordResults[i].getDisplayString();
      for (int k = 0; k < templateResults.length; k++) {
        TemplateProposal curr = templateResults[k];
        if (keyword.equals(curr.getTemplate().getName())) {
          curr.setRelevance(keyWordResults[i].getRelevance());
        }
      }
    }

    // concatenate arrays
    ICompletionProposal[] total = new ICompletionProposal[results.length + templateResults.length];
    System.arraycopy(templateResults, 0, total, 0, templateResults.length);
    System.arraycopy(results, 0, total, templateResults.length, results.length);
    results = total;
    return results;
  }

  private ICompilationUnit findCompilationUnit(ByteArrayOutputStream arrayOutputStream, JETSkeleton skeleton, IPackageFragmentRoot[] roots) throws JavaModelException {
    ICompilationUnit compilationUnit = null;
    for (int i = 0; i < roots.length; i++) {
      IPackageFragmentRoot fragmentRoot = roots[i];
      if (fragmentRoot.getKind() == IPackageFragmentRoot.K_SOURCE) {
        IPackageFragment packageFragment = fragmentRoot.getPackageFragment(skeleton.getPackageName());
        if (packageFragment != null && packageFragment.exists()) {
          compilationUnit = packageFragment.createCompilationUnit(skeleton.getClassName() + ".java", arrayOutputStream
              .toString(), true, new NullProgressMonitor());
          break;
        }
      }
    }
    return compilationUnit;
  }

  private void computeJavaSourceContainer() throws JavaModelException {
    IPackageFragmentRoot[] fragmentRoots = javaProject.getPackageFragmentRoots();
    for (int i = 0; i < fragmentRoots.length; i++) {
      IPackageFragmentRoot fragmentRoot = fragmentRoots[i];
      if (fragmentRoot.getKind() == IPackageFragmentRoot.K_SOURCE) {
        jetNature.setJavaSourceContainer((IContainer) fragmentRoot.getCorrespondingResource());
        break;
      }
    }
  }

  /**
   * Order the given proposals.
   * @param proposals ICompletionProposal[]
   * @return ICompletionProposal[]
   */
  private ICompletionProposal[] order(ICompletionProposal[] proposals) {
    Arrays.sort(proposals, fComparator);
    return proposals;
  }

  public char[] getCompletionProposalAutoActivationCharacters() {
    return fProposalAutoActivationSet;
  }

  public IContextInformation[] computeContextInformation(ITextViewer viewer, int offset) {
    // TODO Auto-generated method stub
    return null;
  }

  public char[] getContextInformationAutoActivationCharacters() {
    // TODO Auto-generated method stub
    return null;
  }

  public String getErrorMessage() {
    // TODO Auto-generated method stub
    return null;
  }

  public IContextInformationValidator getContextInformationValidator() {
    // TODO Auto-generated method stub
    return null;
  }
}