/// buildin Commands and vars
use anyhow::Result;
use std::process::Command;
use std::sync::LazyLock;
use std::{collections::HashMap, iter::zip};
use tower_lsp::lsp_types::{CompletionItem, CompletionItemKind, Documentation, InsertTextFormat};

use crate::consts::TREESITTER_CMAKE_LANGUAGE;
use crate::languageserver::client_support_snippet;
use crate::utils::get_node_content;
use crate::CMakeNodeKinds;

/// convert input text to a snippet, if possible.
fn convert_to_lsp_snippet(key: &str, input: &str) -> String {
    let mut parse = tree_sitter::Parser::new();
    parse.set_language(&TREESITTER_CMAKE_LANGUAGE).unwrap();
    let tree = parse.parse(input, None).unwrap();
    let mut node = tree.root_node().child(0).unwrap();
    if node.kind() != CMakeNodeKinds::NORMAL_COMMAND {
        return input.to_string();
    }
    let mut v: Vec<String> = vec![];
    let mut i = 0;
    node = node.child(2).unwrap();
    if node.kind() != CMakeNodeKinds::ARGUMENT_LIST {
        return input.to_string();
    };
    let source: Vec<&str> = input.split('\n').collect();
    node = node.child(0).unwrap();
    let mut last_position = node.end_position();
    loop {
        if node.kind() == CMakeNodeKinds::ARGUMENT {
            i += 1;
            let start_position = node.start_position();
            let padding = if last_position.row == start_position.row || v.is_empty() {
                "".to_owned()
            } else {
                "\n".to_owned() + &source[start_position.row][0..start_position.column]
            };

            // support at most 9 tab-stops.
            if i < 10 {
                v.push(format!(
                    "{}${{{}:{}}}",
                    padding,
                    i,
                    get_node_content(&source, &node)
                ));
            } else {
                v.push(format!("{}{}", padding, get_node_content(&source, &node)));
            }
            last_position = node.end_position();
        }
        match node.next_sibling() {
            Some(c) => node = c,
            _ => break,
        };
    }
    format!("{}({})", key, v.join(" "))
}

fn gen_buildin_commands(raw_info: &str) -> Result<Vec<CompletionItem>> {
    let re = regex::Regex::new(r"[a-zA-z]+\n-+").unwrap();
    let keys: Vec<_> = re
        .find_iter(raw_info)
        .map(|message| {
            let temp: Vec<&str> = message.as_str().split('\n').collect();
            temp[0]
        })
        .collect();
    let contents: Vec<_> = re.split(raw_info).collect();
    let contents = &contents[1..].to_vec();

    let mut completes = HashMap::new();
    for (key, content) in keys.iter().zip(contents) {
        let small_key = key.to_lowercase();
        let big_key = key.to_uppercase();
        completes.insert(small_key, content.to_string());
        completes.insert(big_key, content.to_string());
    }
    #[cfg(unix)]
    {
        completes.insert(
            "pkg_check_modules".to_string(),
            "please findpackage PkgConfig first".to_string(),
        );
        completes.insert(
            "PKG_CHECK_MODULES".to_string(),
            "please findpackage PkgConfig first".to_string(),
        );
    }

    let client_support_snippet = client_support_snippet();

    Ok(completes
        .iter()
        .map(|(akey, message)| {
            let mut insert_text_format = InsertTextFormat::PLAIN_TEXT;
            let mut insert_text = akey.to_string();
            let mut detail = "Function".to_string();
            let s = format!(r"\n\s+(?P<signature>{}\([^)]*\))", akey);
            let r_match_signature = regex::Regex::new(s.as_str()).unwrap();

            // snippets only work for lower case for now...
            if client_support_snippet
                && insert_text
                    .chars()
                    .all(|c| c.is_ascii_lowercase() || c == '_')
            {
                insert_text = match r_match_signature.captures(message) {
                    Some(m) => {
                        insert_text_format = InsertTextFormat::SNIPPET;
                        detail += " (Snippet)";
                        convert_to_lsp_snippet(akey, m.name("signature").unwrap().as_str())
                    }
                    _ => akey.to_string(),
                }
            };

            CompletionItem {
                label: akey.to_string(),
                kind: Some(CompletionItemKind::FUNCTION),
                detail: Some(detail),
                documentation: Some(Documentation::String(message.to_string())),
                insert_text: Some(insert_text),
                insert_text_format: Some(insert_text_format),
                ..Default::default()
            }
        })
        .collect())
}

fn gen_buildin_variables(raw_info: &str) -> Result<Vec<CompletionItem>> {
    let re = regex::Regex::new(r"[z-zA-z]+\n-+").unwrap();
    let key: Vec<_> = re
        .find_iter(raw_info)
        .map(|message| {
            let temp: Vec<&str> = message.as_str().split('\n').collect();
            temp[0]
        })
        .collect();
    let content: Vec<_> = re.split(raw_info).collect();
    let context = &content[1..];
    Ok(zip(key, context)
        .map(|(akey, message)| CompletionItem {
            label: akey.to_string(),
            kind: Some(CompletionItemKind::VARIABLE),
            detail: Some("Variable".to_string()),
            documentation: Some(Documentation::String(message.to_string())),
            ..Default::default()
        })
        .collect())
}

fn gen_buildin_modules(raw_info: &str) -> Result<Vec<CompletionItem>> {
    let re = regex::Regex::new(r"[z-zA-z]+\n-+").unwrap();
    let key: Vec<_> = re
        .find_iter(raw_info)
        .map(|message| {
            let temp: Vec<&str> = message.as_str().split('\n').collect();
            temp[0]
        })
        .collect();
    let content: Vec<_> = re.split(raw_info).collect();
    let context = &content[1..];
    Ok(zip(key, context)
        .map(|(akey, message)| CompletionItem {
            label: akey.to_string(),
            kind: Some(CompletionItemKind::MODULE),
            detail: Some("Module".to_string()),
            documentation: Some(Documentation::String(message.to_string())),
            ..Default::default()
        })
        .collect())
}

/// CMake build in commands
pub static BUILDIN_COMMAND: LazyLock<Result<Vec<CompletionItem>>> = LazyLock::new(|| {
    let output = Command::new("cmake")
        .arg("--help-commands")
        .output()?
        .stdout;
    let temp = String::from_utf8_lossy(&output);
    gen_buildin_commands(&temp)
});

/// cmake buildin vars
pub static BUILDIN_VARIABLE: LazyLock<Result<Vec<CompletionItem>>> = LazyLock::new(|| {
    let output = Command::new("cmake")
        .arg("--help-variables")
        .output()?
        .stdout;
    let temp = String::from_utf8_lossy(&output);
    gen_buildin_variables(&temp)
});

/// Cmake buildin modules
pub static BUILDIN_MODULE: LazyLock<Result<Vec<CompletionItem>>> = LazyLock::new(|| {
    let output = Command::new("cmake").arg("--help-modules").output()?.stdout;
    let temp = String::from_utf8_lossy(&output);
    gen_buildin_modules(&temp)
});

#[cfg(test)]
mod tests {
    use std::iter::zip;

    use crate::complete::buildin::{gen_buildin_modules, gen_buildin_variables};

    use super::gen_buildin_commands;
    #[test]
    fn tst_regex() {
        let re = regex::Regex::new(r"-+").unwrap();
        assert!(re.is_match("---------"));
        assert!(re.is_match("-------------------"));
        let temp = "javascrpt---------it is";
        let splits: Vec<_> = re.split(temp).collect();
        let aftersplit = vec!["javascrpt", "it is"];
        for (split, after) in zip(splits, aftersplit) {
            assert_eq!(split, after);
        }
    }

    #[test]
    fn tst_cmake_command_buildin() {
        // NOTE: In case the command fails, ignore test
        let output = include_str!("../../assert/cmake_help_commands.txt");

        let output = gen_buildin_commands(&output);

        assert!(output.is_ok());
    }

    #[test]
    fn tst_cmake_variables_buildin() {
        // NOTE: In case the command fails, ignore test
        let output = include_str!("../../assert/cmake_help_variables.txt");

        let output = gen_buildin_variables(&output);

        assert!(output.is_ok());
    }

    #[test]
    fn tst_cmake_modules_buildin() {
        // NOTE: In case the command fails, ignore test
        let output = include_str!("../../assert/cmake_help_commands.txt");

        let output = gen_buildin_modules(&output);

        assert!(output.is_ok());
    }
}
