/* Copyright (c) <2003-2022> <Newton Game Dynamics>
* 
* This software is provided 'as-is', without any express or implied
* warranty. In no event will the authors be held liable for any damages
* arising from the use of this software.
* 
* Permission is granted to anyone to use this software for any purpose,
* including commercial applications, and to alter it and redistribute it
* freely
*/

#include "ndSandboxStdafx.h"
#include "ndSkyBox.h"
#include "ndDemoMesh.h"
#include "ndUIEntity.h"
#include "ndDemoCamera.h"
#include "ndLoadFbxMesh.h"
#include "ndPhysicsUtils.h"
#include "ndPhysicsWorld.h"
#include "ndMakeStaticMap.h"
#include "ndDemoEntityNotify.h"
#include "ndDemoEntityManager.h"
#include "ndDemoInstanceEntity.h"

namespace ndSimpleRobot
{
	class ndDefinition
	{
		public:
		enum ndJointType
		{
			m_root,
			m_hinge,
			m_slider,
			m_effector
		};

		char m_boneName[32];
		ndJointType m_type;
		ndFloat32 m_mass;
		ndFloat32 m_minLimit;
		ndFloat32 m_maxLimit;
	};

	static ndDefinition jointsDefinition[] =
	{
		{ "base", ndDefinition::m_root, 100.0f, 0.0f, 0.0f},
		{ "base_rotator", ndDefinition::m_hinge, 50.0f, -1.0e10f, 1.0e10f},
		{ "arm_0", ndDefinition::m_hinge , 5.0f, -140.0f * ndDegreeToRad, 1.0f * ndDegreeToRad},
		{ "arm_1", ndDefinition::m_hinge , 5.0f, -30.0f * ndDegreeToRad, 120.0f * ndDegreeToRad},
		{ "arm_2", ndDefinition::m_hinge , 5.0f, -1.0e10f, 1.0e10f},
		{ "arm_3", ndDefinition::m_hinge , 3.0f, -1.0e10f, 1.0e10f},
		{ "arm_4", ndDefinition::m_hinge , 2.0f, -1.0e10f, 1.0e10f},
		{ "gripperLeft", ndDefinition::m_slider , 1.0f, -0.2f, 0.03f},
		{ "gripperRight", ndDefinition::m_slider , 1.0f, -0.2f, 0.03f},
		{ "effector", ndDefinition::m_effector , 0.0f, 0.0f, 0.0f},
	};

	class ndIndustrialRobot : public ndModel
	{
		public:
		D_CLASS_REFLECTION(ndSimpleRobot::ndIndustrialRobot);

		ndIndustrialRobot(ndDemoEntityManager* const scene, ndDemoEntity* const robotMesh, const ndMatrix& location)
			:ndModel()
			,m_rootBody(nullptr)
			,m_leftGripper(nullptr)
			,m_rightGripper(nullptr)
			,m_effector(nullptr)
			,m_effectorOffset(ndVector::m_wOne)
			,m_x(0.0f)
			,m_y(0.0f)
			,m_azimuth(0.0f)
			,m_gripperPosit(0.0f)
			,m_pitch(0.0f)
			,m_yaw(0.0f)
			,m_roll(0.0f)
		{
			// make a clone of the mesh and add it to the scene
			ndDemoEntity* const rootEntity = robotMesh->CreateClone();
			scene->AddEntity(rootEntity);
			ndWorld* const world = scene->GetWorld();
			
			// find the floor location 
			ndMatrix matrix(location);
			ndVector floor(FindFloor(*world, matrix.m_posit + ndVector(0.0f, 100.0f, 0.0f, 0.0f), 200.0f));
			matrix.m_posit.m_y = floor.m_y;
			
			//matrix.m_posit.m_y += 1.0f;
			rootEntity->ResetMatrix(matrix);
			
			// add the root body
			m_rootBody = CreateBodyPart(scene, rootEntity, jointsDefinition[0].m_mass, nullptr);
			m_bodyArray.PushBack(m_rootBody);
			
			ndFixSizeArray<ndDemoEntity*, 32> childEntities;
			ndFixSizeArray<ndBodyDynamic*, 32> parentBone;
			
			ndInt32 stack = 0;
			for (ndDemoEntity* child = rootEntity->GetFirstChild(); child; child = child->GetNext())
			{
				childEntities[stack] = child;
				parentBone[stack] = m_rootBody;
				stack++;
			}
			
			const ndInt32 definitionCount = ndInt32(sizeof(jointsDefinition) / sizeof(jointsDefinition[0]));
			while (stack)
			{
				stack--;
				ndBodyDynamic* parentBody = parentBone[stack];
				ndDemoEntity* const childEntity = childEntities[stack];
			
				const char* const name = childEntity->GetName().GetStr();
				for (ndInt32 i = 0; i < definitionCount; ++i)
				{
					const ndDefinition& definition = jointsDefinition[i];
					if (!strcmp(definition.m_boneName, name))
					{
						ndTrace(("name: %s\n", name));
						if (definition.m_type == ndDefinition::m_hinge)
						{
							ndBodyDynamic* const childBody = CreateBodyPart(scene, childEntity, definition.m_mass, parentBody);
							m_bodyArray.PushBack(childBody);
							const ndMatrix pivotMatrix(childBody->GetMatrix());
							ndJointHinge* const hinge = new ndJointHinge(pivotMatrix, childBody, parentBody);
							hinge->SetLimits(definition.m_minLimit, definition.m_maxLimit);
							m_jointArray.PushBack(hinge);

							ndSharedPtr<ndJointBilateralConstraint> hingePtr(hinge);
							world->AddJoint(hingePtr);
							parentBody = childBody;
						}
						else if (definition.m_type == ndDefinition::m_slider)
						{
							ndBodyDynamic* const childBody = CreateBodyPart(scene, childEntity, definition.m_mass, parentBody);
							m_bodyArray.PushBack(childBody);
			
							const ndMatrix pivotMatrix(childBody->GetMatrix());
							ndJointSlider* const slider = new ndJointSlider(pivotMatrix, childBody, parentBody);
							slider->SetLimits(definition.m_minLimit, definition.m_maxLimit);
							slider->SetAsSpringDamper(0.005f, 2000.0f, 200.0f);
			
							if (!strstr(definition.m_boneName, "Left"))
							{
								m_leftGripper = slider;
							}
							else
							{
								m_rightGripper = slider;
							}
							ndSharedPtr<ndJointBilateralConstraint> sliderPtr(slider);
							world->AddJoint(sliderPtr);
							
							parentBody = childBody;
						}
						else
						{
							ndBodyDynamic* const childBody = parentBody;
			
							const ndMatrix pivotFrame(rootEntity->Find("referenceFrame")->CalculateGlobalMatrix());
							const ndMatrix effectorFrame(childEntity->CalculateGlobalMatrix());
							m_effector = new ndIk6DofEffector(effectorFrame, pivotFrame, childBody, m_rootBody);
			
							m_effectorOffset = m_effector->GetOffsetMatrix().m_posit;
			
							ndFloat32 relaxation = 0.003f;
							m_effector->EnableRotationAxis(ndIk6DofEffector::m_shortestPath);
							m_effector->SetLinearSpringDamper(relaxation, 1500.0f, 100.0f);
							m_effector->SetAngularSpringDamper(relaxation, 1500.0f, 100.0f);
							m_effector->SetMaxForce(10000.0f);
							m_effector->SetMaxTorque(10000.0f);
			
							// the effector is part of the rig
							ndSharedPtr<ndJointBilateralConstraint> effectorPtr(m_effector);
							world->AddJoint(effectorPtr);
						}
						break;
					}
				}
			
				for (ndDemoEntity* child = childEntity->GetFirstChild(); child; child = child->GetNext())
				{
					childEntities[stack] = child;
					parentBone[stack] = parentBody;
					stack++;
				}
			}
		}

		ndIndustrialRobot(const ndLoadSaveBase::ndLoadDescriptor& desc)
			:ndModel(ndLoadSaveBase::ndLoadDescriptor(desc))
			,m_rootBody(nullptr)
			,m_leftGripper(nullptr)
			,m_rightGripper(nullptr)
			,m_effector(nullptr)
			,m_effectorOffset(ndVector::m_wOne)
			,m_x(0.0f)
			,m_y(0.0f)
			,m_azimuth(0.0f)
			,m_gripperPosit(0.0f)
			,m_pitch(0.0f)
			,m_yaw(0.0f)
			,m_roll(0.0f)
		{
			const nd::TiXmlNode* const modelRootNode = desc.m_rootNode;

			const nd::TiXmlNode* const bodies = modelRootNode->FirstChild("bodies");
			for (const nd::TiXmlNode* node = bodies->FirstChild(); node; node = node->NextSibling())
			{
				ndInt32 hashId;
				const nd::TiXmlElement* const element = (nd::TiXmlElement*) node;
				element->Attribute("int32", &hashId);
				ndBodyLoaderCache::ndNode* const bodyNode = desc.m_bodyMap->Find(hashId);

				ndBody* const body = (ndBody*)bodyNode->GetInfo();
				m_bodyArray.PushBack(body->GetAsBodyDynamic());
			}

			const nd::TiXmlNode* const joints = modelRootNode->FirstChild("joints");
			for (const nd::TiXmlNode* node = joints->FirstChild(); node; node = node->NextSibling())
			{
				ndInt32 hashId;
				const nd::TiXmlElement* const element = (nd::TiXmlElement*) node;
				element->Attribute("int32", &hashId);
				ndJointLoaderCache::ndNode* const jointNode = desc.m_jointMap->Find(hashId);

				ndJointBilateralConstraint* const joint = (ndJointBilateralConstraint*)jointNode->GetInfo();
				m_jointArray.PushBack((ndJointHinge*)joint);
			}

			// load root body
			ndBodyLoaderCache::ndNode* const rootBodyNode = desc.m_bodyMap->Find(xmlGetInt(modelRootNode, "rootBodyHash"));
			ndBody* const rootbody = (ndBody*)rootBodyNode->GetInfo();
			m_rootBody = rootbody->GetAsBodyDynamic();

			// load effector joint
			const nd::TiXmlNode* const endEffectorNode = modelRootNode->FirstChild("endEffector");
			if (xmlGetInt(endEffectorNode, "hasEffector"))
			{
				ndAssert(0);
				//ndBodyLoaderCache::ndNode* const effectorBodyNode0 = desc.m_bodyMap->Find(xmlGetInt(endEffectorNode, "body0Hash"));
				//ndBodyLoaderCache::ndNode* const effectorBodyNode1 = desc.m_bodyMap->Find(xmlGetInt(endEffectorNode, "body1Hash"));
				//
				//ndBody* const body0 = (ndBody*)effectorBodyNode0->GetInfo();
				//ndBody* const body1 = (ndBody*)effectorBodyNode1->GetInfo();
				//ndAssert(body1 == m_rootBody);
				//
				//const ndMatrix pivotMatrix(body0->GetMatrix());
				//m_effector = new ndIk6DofEffector(pivotMatrix, body0->GetAsBodyDynamic(), body1->GetAsBodyDynamic());
				//m_effector->EnableRotationAxis(ndIk6DofEffector::m_shortestPath);
				//m_effector->SetMode(true, true);
				//
				//ndFloat32 regularizer;
				//ndFloat32 springConst;
				//ndFloat32 damperConst;
				//
				//m_effector->GetLinearSpringDamper(regularizer, springConst, damperConst);
				//m_effector->SetLinearSpringDamper(regularizer * 0.5f, springConst * 10.0f, damperConst * 10.0f);
				//
				//m_effector->GetAngularSpringDamper(regularizer, springConst, damperConst);
				//m_effector->SetAngularSpringDamper(regularizer * 0.5f, springConst * 10.0f, damperConst * 10.0f);
			}
		}

		~ndIndustrialRobot()
		{
		}

		void Save(const ndLoadSaveBase::ndSaveDescriptor& desc) const
		{
			nd::TiXmlElement* const modelRootNode = new nd::TiXmlElement(ClassName());
			desc.m_rootNode->LinkEndChild(modelRootNode);
			modelRootNode->SetAttribute("hashId", desc.m_nodeNodeHash);
			ndModel::Save(ndLoadSaveBase::ndSaveDescriptor(desc, modelRootNode));

			// save all bodies.
			nd::TiXmlElement* const bodiesNode = new nd::TiXmlElement("bodies");
			modelRootNode->LinkEndChild(bodiesNode);
			for (ndInt32 i = 0; i < m_bodyArray.GetCount(); ++i)
			{
				nd::TiXmlElement* const paramNode = new nd::TiXmlElement("body");
				bodiesNode->LinkEndChild(paramNode);

				ndTree<ndInt32, const ndBodyKinematic*>::ndNode* const bodyPartNode = desc.m_bodyMap->Insert(desc.m_bodyMap->GetCount(), m_bodyArray[i]);
				paramNode->SetAttribute("int32", bodyPartNode->GetInfo());
			}

			// save all joints
			nd::TiXmlElement* const jointsNode = new nd::TiXmlElement("joints");
			modelRootNode->LinkEndChild(jointsNode);
			for (ndInt32 i = 0; i < m_jointArray.GetCount(); ++i)
			{
				nd::TiXmlElement* const paramNode = new nd::TiXmlElement("joint");
				jointsNode->LinkEndChild(paramNode);

				ndTree<ndInt32, const ndJointBilateralConstraint*>::ndNode* const jointPartNode = desc.m_jointMap->Insert(desc.m_jointMap->GetCount(), m_jointArray[i]);
				paramNode->SetAttribute("int32", jointPartNode->GetInfo());
			}

			// add sliders
			{
				nd::TiXmlElement* const paramNode = new nd::TiXmlElement("joint");
				jointsNode->LinkEndChild(paramNode);

				ndTree<ndInt32, const ndJointBilateralConstraint*>::ndNode* const jointPartNode = desc.m_jointMap->Insert(desc.m_jointMap->GetCount(), m_leftGripper);
				paramNode->SetAttribute("int32", jointPartNode->GetInfo());
			}

			// add sliders
			{
				nd::TiXmlElement* const paramNode = new nd::TiXmlElement("joint");
				jointsNode->LinkEndChild(paramNode);

				ndTree<ndInt32, const ndJointBilateralConstraint*>::ndNode* const jointPartNode = desc.m_jointMap->Insert(desc.m_jointMap->GetCount(), m_rightGripper);
				paramNode->SetAttribute("int32", jointPartNode->GetInfo());
			}

			// indicate which body is the root
			xmlSaveParam(modelRootNode, "rootBodyHash", desc.m_bodyMap->Find(m_rootBody)->GetInfo());

			// save end effector info
			nd::TiXmlElement* const endEffectorNode = new nd::TiXmlElement("endEffector");
			modelRootNode->LinkEndChild(endEffectorNode);

			xmlSaveParam(endEffectorNode, "hasEffector", m_effector ? 1 : 0);
			if (m_effector)
			{
				// save the effector joint
				nd::TiXmlElement* const paramNode = new nd::TiXmlElement("joint");
				jointsNode->LinkEndChild(paramNode);
				ndTree<ndInt32, const ndJointBilateralConstraint*>::ndNode* const jointPartNode = desc.m_jointMap->Insert(desc.m_jointMap->GetCount(), m_effector);
				paramNode->SetAttribute("int32", jointPartNode->GetInfo());

				// and the connection
				ndTree<ndInt32, const ndBodyKinematic*>::ndNode* const effectBody0 = desc.m_bodyMap->Find(m_effector->GetBody0());
				ndTree<ndInt32, const ndBodyKinematic*>::ndNode* const effectBody1 = desc.m_bodyMap->Find(m_effector->GetBody1());
				xmlSaveParam(endEffectorNode, "body0Hash", effectBody0->GetInfo());
				xmlSaveParam(endEffectorNode, "body1Hash", effectBody1->GetInfo());
			}
		}

		ndBodyDynamic* CreateBodyPart(ndDemoEntityManager* const scene, ndDemoEntity* const entityPart, ndFloat32 mass, ndBodyDynamic* const parentBone)
		{
			ndSharedPtr<ndShapeInstance> shapePtr(entityPart->CreateCollisionFromChildren());
			ndShapeInstance* const shape = *shapePtr;
			ndAssert(shape);

			// create the rigid body that will make this body
			ndMatrix matrix(entityPart->CalculateGlobalMatrix());

			ndBodyKinematic* const body = new ndBodyDynamic();
			body->SetMatrix(matrix);
			body->SetCollisionShape(*shape);
			body->SetMassMatrix(mass, *shape);
			body->SetNotifyCallback(new ndDemoEntityNotify(scene, entityPart, parentBone));

			// add body to the world
			ndSharedPtr<ndBody> bodyPtr(body);
			scene->GetWorld()->AddBody(bodyPtr);

			return body->GetAsBodyDynamic();
		}

		ndBodyDynamic* GetRoot() const
		{
			return m_rootBody;
		}

		void Debug(ndConstraintDebugCallback& context) const
		{
			if (m_effector)
			{
				((ndJointBilateralConstraint*)m_effector)->DebugJoint(context);
			}
		}

		void PostUpdate(ndWorld* const world, ndFloat32 timestep)
		{
			ndModel::PostUpdate(world, timestep);
		}

		void PostTransformUpdate(ndWorld* const world, ndFloat32 timestep)
		{
			ndModel::PostTransformUpdate(world, timestep);
		}

		void Update(ndWorld* const world, ndFloat32 timestep)
		{
			ndModel::Update(world, timestep);
			if (m_effector)
			{
				// apply target position collected by control panel
				ndMatrix targetMatrix(
					ndRollMatrix(90.0f * ndDegreeToRad) *
					ndPitchMatrix(m_pitch * ndDegreeToRad) *
					ndYawMatrix(m_yaw * ndDegreeToRad) *
					ndRollMatrix(m_roll * ndDegreeToRad) *
					ndRollMatrix(-90.0f * ndDegreeToRad));

				ndVector localPosit(m_x, m_y, 0.0f, 0.0f);
				const ndMatrix aximuthMatrix(ndYawMatrix(m_azimuth * ndDegreeToRad));
				targetMatrix.m_posit = aximuthMatrix.TransformVector(m_effectorOffset + localPosit);

				m_effector->SetOffsetMatrix(targetMatrix);
				m_leftGripper->SetOffsetPosit(-m_gripperPosit * 0.5f);
				m_rightGripper->SetOffsetPosit(-m_gripperPosit * 0.5f);
			}
		}

		ndBodyDynamic* m_rootBody;
		ndJointSlider* m_leftGripper;
		ndJointSlider* m_rightGripper;
		ndIk6DofEffector* m_effector;
		ndFixSizeArray<ndBodyDynamic*, 16> m_bodyArray;
		ndFixSizeArray<ndJointBilateralConstraint*, 16> m_jointArray;
		ndVector m_effectorOffset;
		ndReal m_x;
		ndReal m_y;
		ndReal m_azimuth;
		ndReal m_gripperPosit;
		ndReal m_pitch;
		ndReal m_yaw;
		ndReal m_roll;
	};
	D_CLASS_REFLECTION_IMPLEMENT_LOADER(ndSimpleRobot::ndIndustrialRobot);

	class ndRobotUI : public ndUIEntity
	{
		public:
		ndRobotUI(ndDemoEntityManager* const scene, ndIndustrialRobot* const robot)
			:ndUIEntity(scene)
			,m_robot(robot)
		{
		}

		~ndRobotUI()
		{
		}

		virtual void RenderUI()
		{
		}

		virtual void RenderHelp()
		{
			ndVector color(1.0f, 1.0f, 0.0f, 0.0f);
			m_scene->Print(color, "Control panel");

			bool change = false;
			ImGui::Text("position x");
			change = change | ImGui::SliderFloat("##x", &m_robot->m_x, 0.0f, 5.0f);
			ImGui::Text("position y");
			change = change | ImGui::SliderFloat("##y", &m_robot->m_y, -2.5f, 2.0f);
			ImGui::Text("azimuth");
			change = change | ImGui::SliderFloat("##azimuth", &m_robot->m_azimuth, -180.0f, 180.0f);
			
			ImGui::Text("gripper");
			change = change | ImGui::SliderFloat("##gripper", &m_robot->m_gripperPosit, -0.2f, 0.4f);
			
			ImGui::Text("pitch");
			change = change | ImGui::SliderFloat("##pitch", &m_robot->m_pitch, -180.0f, 180.0f);
			ImGui::Text("yaw");
			change = change | ImGui::SliderFloat("##yaw", &m_robot->m_yaw, -180.0f, 180.0f);
			ImGui::Text("roll");
			change = change | ImGui::SliderFloat("##roll", &m_robot->m_roll, -180.0f, 180.0f);
			
			if (change)
			{
				m_robot->m_rootBody->SetSleepState(false);
			}
		}

		ndIndustrialRobot* m_robot;
	};
};

using namespace ndSimpleRobot;
void ndSimpleIndustrialRobot (ndDemoEntityManager* const scene)
{
	// build a floor
	BuildFloorBox(scene, ndGetIdentityMatrix());
	
	ndVector origin1(0.0f, 0.0f, 0.0f, 1.0f);
	ndDemoEntity* const robotEntity = ndDemoEntity::LoadFbx("robot.fbx", scene);
	
	ndWorld* const world = scene->GetWorld();
	ndMatrix matrix(ndYawMatrix(-90.0f * ndDegreeToRad));
	ndIndustrialRobot* const robot = new ndIndustrialRobot(scene, robotEntity, matrix);
	scene->SetSelectedModel(robot);

	ndSharedPtr<ndModel> robotPtr(robot);
	ndSharedPtr<ndJointBilateralConstraint> fixJoint(new ndJointFix6dof(robot->GetRoot()->GetMatrix(), robot->GetRoot(), world->GetSentinelBody()));
	world->AddModel(robotPtr);
	world->AddJoint (fixJoint);
	
	ndRobotUI* const robotUI = new ndRobotUI(scene, robot);
	ndSharedPtr<ndUIEntity> robotUIPtr(robotUI);
	scene->Set2DDisplayRenderFunction(robotUIPtr);
	
	//matrix.m_posit.m_x += 2.0f;
	//matrix.m_posit.m_z -= 2.0f;
	//scene->GetWorld()->AddModel(new ndIndustrialRobot(scene, robotEntity, matrix));
	//ndMatrix location(matrix);
	//location = ndRollMatrix(-65 * ndDegreeToRad) * location;
	//location.m_posit.m_x += 1.5f;
	//location.m_posit.m_y += 1.5f;
	//location.m_posit.m_z += 1.5f;
	//AddBox(scene, location, 5.0f, 0.5f, 3.0f, 4.0f);
	//location.m_posit.m_z += 1.5f;
	//AddBox(scene, location, 5.0f, 0.5f, 3.0f, 4.0f);
	
	delete robotEntity;
	
	matrix.m_posit.m_x -= 6.0f;
	matrix.m_posit.m_y += 2.0f;
	matrix.m_posit.m_z += 6.0f;
	ndQuaternion rotation(ndVector(0.0f, 1.0f, 0.0f, 0.0f), 45.0f * ndDegreeToRad);
	scene->SetCameraMatrix(rotation, matrix.m_posit);
}
