// Copyright 2022 the V8 project authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "src/base/macros.h"
#include "src/base/numbers/fast-dtoa.h"
#include "src/base/vector.h"
#include "third_party/google_benchmark/src/include/benchmark/benchmark.h"

using v8::base::FAST_DTOA_PRECISION;
using v8::base::FAST_DTOA_SHORTEST;
using v8::base::kFastDtoaMaximalLength;
using v8::base::Vector;

// This is a dump from a benchmark (MotionMark suits).
// It contains a large number of nontrivial doubles,
// including the ones that fail the fast-dtoa algorithm.
constexpr const double kTestDoubles[4096] = {
    468.1588596224, 480.3152423710, 523.3453890260, 570.1679798156,
    190.0465907999, 562.4923651186, 373.9153690274, 570.5984547204,
    515.4506822597, 308.7429754814, 176.5603162753, 43.9027116524,
    481.3434789318, 41.0993040511,  461.0124155912, 228.7679668768,
    491.6391940140, 733.0830583525, 227.6475280451, 493.0394209479,
    279.5408924144, 735.1125461811, 492.0691872806, 737.3663743301,
    87.1932548647,  762.5608311158, 300.9207923423, 246.1835506766,
    181.2835066099, 691.5725161284, 375.5178018165, 509.0501544704,
    207.1629680427, 371.9653871617, 310.2547970842, 279.4268876632,
    272.2559833257, 354.8581094830, 292.4814681094, 572.6136458742,
    291.1816687366, 835.3018571642, 117.5735765227, 584.8558183250,
    375.5703242000, 96.1422415615,  465.7228320626, 230.3613560100,
    558.1811795368, 586.1937948580, 193.3071465316, 164.7315231297,
    550.2162216151, 205.9741458774, 484.2569807606, 776.4126345078,
    374.4623562300, 290.1958843022, 453.9585512747, 465.4893101328,
    548.2565749399, 237.3136547446, 508.7985270223, 524.6502722544,
    193.4088935622, 77.9520729860,  160.5456417440, 54.6766152489,
    780.4914417607, 426.0727237702, 516.4995632726, 467.0438681707,
    827.7248524368, 275.1848233277, 506.1141300144, 73.0874965955,
    452.7129190767, 458.2850162522, 649.4299442894, 174.0508265151,
    270.2334743447, 162.0554894773, 844.6592777163, 277.4507392173,
    245.2313072573, 213.7873144789, 438.9344755736, 49.8126353642,
    187.4533692831, 531.3201674381, 425.8655536173, 379.4700465341,
    791.5863745783, 487.3973446329, 402.3151309389, 689.4106297947,
    223.7827699703, 846.7327281835, 237.1972576348, 614.7105084412,
    250.9736372060, 183.0601289549, 441.3473549917, 730.2877659622,
    176.3644984393, 372.8873484144, 343.4431735106, 866.6693864452,
    550.6362287984, 308.0283033251, 452.0509174680, 756.5892672550,
    260.9771943310, 130.1919787891, 476.7450943186, 440.7273213825,
    292.7602563236, 356.8079870938, 291.4262464090, 80.4305399214,
    703.7380547633, 261.3531254844, 215.4715027252, 431.7450361498,
    26.0419129206,  308.5045893580, 588.9014064061, 463.5525083473,
    821.7421101484, 228.4247100458, 766.0971221778, 339.4377738174,
    412.6341570736, 120.4403656706, 746.5716149113, 556.4490946118,
    797.5920347736, 241.5991630926, 612.1756970343, 229.2432469916,
    436.6797336511, 314.9449072017, 678.8403882783, 265.8133004467,
    284.4977102105, 443.3688720197, 751.8110258991, 288.6424820345,
    826.3144706325, 511.0279680708, 266.0506115043, 468.0653613444,
    539.4045651487, 468.7310158448, 81.0144695011,  400.0658987168,
    775.2842887593, 214.8558022547, 637.9186211297, 97.4338479295,
    202.7853649035, 152.1996706865, 637.6331366756, 292.7410837653,
    744.7716440842, 224.1139979479, 571.5668345547, 360.4495549501,
    782.1958542804, 188.0105462824, 38.0868154957,  465.4173187423,
    494.2204254255, 62.0640839692,  193.0405623360, 69.3528878075,
    356.0777562379, 661.2374952557, 228.0128880273, 325.3917825379,
    454.5339908012, 486.1457130016, 126.7946914065, 78.4242259984,
    266.2578038947, 574.9997956419, 150.2799424673, 808.8567313210,
    129.1252583358, 884.5000000000, 486.5468173868, 119.5950037048,
    488.4131881457, 584.3793369729, 84.9875334191,  203.6093123783,
    182.2859868258, 494.7116078570, 458.2190888610, 61.4774560550,
    402.3898069806, 728.1262779382, 347.0235583124, 829.6507381468,
    471.2398388466, 169.7708006324, 479.0151953503, 652.4946064187,
    348.6151379498, 79.7056834059,  243.6418719160, 722.8087194220,
    466.1883324660, 790.3884648281, 581.5648934640, 150.8295394505,
    462.6093111947, 42.2227945536,  179.4090043741, 372.8065515741,
    230.9919588033, 389.1472213205, 436.2594879588, 709.6601569093,
    88.1027017046,  22.5000000000,  403.3002883813, 833.5137523583,
    414.1482937256, 416.9911612710, 42.1947903320,  346.9305632705,
    540.2680792693, 573.6009903024, 353.5650585340, 703.3171894339,
    167.7476576376, 474.5765114232, 278.1135582025, 255.8693864644,
    360.8026986553, 84.9254816069,  295.6691419007, 259.6184858094,
    512.4030535233, 498.9198620968, 758.7456517081, 244.9206499767,
    16.7980179552,  75.8480444114,  135.1264626047, 438.8486465591,
    406.8157704753, 184.2350004046, 182.0958427683, 466.9882405134,
    508.4265398356, 261.8592275943, 559.6191545164, 197.7835796500,
    682.2163750726, 308.2233956586, 813.6172917328, 465.0575642925,
    88.8299171865,  135.5335850727, 254.5339945217, 463.6714454391,
    812.0230595122, 258.7931153396, 677.8951299782, 232.0559746924,
    34.8982603843,  519.1818744522, 527.4875215610, 495.0556590611,
    127.2060663755, 190.7010077010, 547.1644761968, 180.0473016614,
    676.3706880534, 461.8524783928, 213.8277535164, 97.8010744851,
    850.2959113846, 360.7417196351, 277.9543674996, 178.5236196748,
    90.4217541687,  198.3290708395, 277.5007351598, 350.7035905150,
    67.1082096846,  396.5931702061, 877.9275197004, 462.8396392321,
    77.0916598052,  420.1914155606, 724.3312211248, 180.8459986492,
    882.2705245878, 193.7235160745, 59.0411850211,  331.4489310045,
    91.5465590356,  365.0733424801, 227.9788120772, 250.7546503651,
    240.7035174334, 135.0715690820, 255.9093611959, 103.0732465910,
    583.5000000000, 467.7582573911, 253.8132344735, 382.0047947124,
    286.7705161157, 430.8770132382, 557.3196136475, 467.2204275684,
    47.9794737917,  137.7357904719, 232.7129390863, 329.7287725700,
    185.8077710098, 548.2746613969, 288.3794649303, 276.1350352103,
    434.9596003850, 245.3556661520, 226.2046116105, 529.3626355976,
    71.5386842779,  748.1890120862, 526.6588344006, 439.8470831188,
    432.0930332308, 665.2821626850, 261.5584726101, 572.4952718185,
    354.2129324571, 275.2556654028, 343.5661185854, 439.0880701817,
    111.4999318329, 470.3676929109, 456.3160712012, 569.8699369857,
    47.2841506100,  739.1812859068, 19.6172667580,  458.7706302263,
    219.0526785488, 391.5257563787, 74.8714927980,  463.4834484495,
    775.0867976781, 146.5856110625, 491.7471713388, 138.0562846286,
    597.7429202676, 352.6364405246, 192.0631840033, 177.7225067993,
    209.3954931344, 269.4641946244, 57.2247292912,  568.4658236656,
    857.2830795107, 529.4285984529, 335.0138259073, 114.7556280053,
    356.5974421783, 464.3530212797, 120.0515315526, 233.6829120313,
    489.9658934763, 125.4273019459, 732.9703056942, 572.7137286249,
    612.6597962940, 406.8909817441, 34.6886847306,  443.9002144852,
    627.9462826211, 471.2208432799, 278.5056764723, 377.5707253631,
    450.1937072957, 189.6392490642, 736.7944841153, 27.8749537844,
    523.6884842045, 251.6608864242, 743.5831047430, 504.9621433452,
    809.4955333103, 273.2013142694, 849.0200482558, 431.0941276276,
    146.4527203661, 219.8787653843, 326.0590586927, 506.0773400813,
    750.6237219131, 529.2881719903, 142.8579560342, 213.2316477743,
    688.4728314852, 535.5787206238, 461.6623032178, 475.0944398827,
    175.5966807158, 335.1151819121, 471.2136094302, 467.8323086700,
    396.5173883633, 250.8460236858, 706.4894229601, 153.0950909286,
    788.5334476657, 478.3854390729, 139.3264699593, 533.1816475667,
    648.4910609145, 327.0099230131, 155.3251635863, 266.0311408651,
    240.0511731779, 810.6179678204, 329.2489588901, 201.5497396899,
    470.5755719123, 174.3237777035, 428.4787954110, 102.4009615461,
    337.8621567202, 430.4749120479, 677.5211397605, 311.5311951225,
    374.5576534741, 289.8332025061, 662.2893894385, 452.2515202318,
    399.3249017949, 481.9510838776, 61.3584044335,  341.8616690931,
    513.1620612293, 184.3003039998, 264.3987906182, 165.9855167055,
    729.1875250790, 520.9548057848, 205.1872074459, 290.9794891538,
    186.2846129908, 507.0961311202, 787.7005538004, 452.4821885632,
    461.5744575612, 474.8184657202, 235.1909837669, 577.4070807754,
    261.8480066013, 100.7294297995, 38.9045159166,  299.8782935684,
    675.8971036095, 114.4202522220, 641.0206514540, 561.0036522169,
    258.3074994420, 228.3066891185, 592.4152753285, 68.8376708090,
    578.3508909872, 91.5645298205,  316.1097113152, 501.3930207871,
    138.0715853373, 766.5588975894, 294.7349323019, 489.4035963387,
    261.2447152829, 87.0689345250,  190.5698862432, 49.1672796667,
    227.7897786652, 568.9529130837, 272.7899442553, 485.8919869450,
    371.5790500621, 285.6709471653, 258.0152365968, 803.4392535146,
    533.3594896129, 223.6525494198, 548.3431153405, 866.5712777370,
    167.6691830755, 538.2955434401, 71.3453181470,  220.4777226597,
    576.6866092503, 205.1444302019, 182.4833507986, 95.6127185811,
    125.9495877082, 736.9705595895, 295.4486766628, 764.3920529565,
    552.9833484801, 78.9335944363,  116.3325984505, 201.5321001957,
    29.7625281794,  533.8622328568, 498.4066442650, 382.0897436055,
    549.3293274395, 740.3585239321, 301.7123158343, 592.3586141814,
    260.2621594400, 684.8418391077, 278.8280194020, 800.3029298065,
    298.4474494276, 329.0443873758, 226.7235313434, 406.2821026208,
    383.2683666591, 756.1296472501, 407.0649153487, 209.3483952462,
    444.0504015055, 71.2084800261,  342.5365403474, 516.4893129090,
    340.5325402108, 470.4869220887, 321.7571713083, 142.5171713727,
    338.1453265880, 382.0030245468, 384.2816594972, 545.1176011152,
    170.0987264769, 323.8026240543, 337.6365363991, 433.7411830353,
    58.9071151611,  200.3050385152, 778.5624426703, 500.0402479571,
    614.5538989565, 466.6254749803, 776.7269195474, 279.8259676875,
    705.9453211331, 463.9647624701, 299.2772177283, 347.9224715474,
    128.0395006953, 169.5937409096, 150.2891311846, 314.2536711159,
    652.5786055030, 366.0362876005, 606.1980299473, 500.7979020236,
    738.8533700241, 135.3760081455, 180.8784060345, 369.9019068674,
    714.7990591272, 509.9282102240, 550.3499421783, 468.1426558391,
    708.4982398855, 208.2682402528, 862.5315972697, 507.3673299905,
    18.1837093974,  306.4385916327, 278.7680825049, 466.7728485341,
    572.7996353839, 51.4281915470,  833.9190829209, 299.4227212449,
    576.9270604306, 363.0907760628, 447.7349074443, 283.3384815038,
    576.0852243676, 407.1559358485, 100.8463616601, 531.2206661099,
    416.0873967507, 520.8621499277, 308.5074777197, 515.8182766043,
    209.5000000000, 281.5000000000, 301.2559464406, 28.2901967445,
    831.8578966754, 527.5110229639, 445.3392869505, 133.6636777893,
    285.7982464996, 426.7475115510, 820.4371608704, 385.8079181592,
    585.7686680200, 336.7421089350, 872.7675647593, 446.3788737769,
    43.7120670459,  408.8272032431, 668.3648205564, 257.3771685896,
    679.5836110410, 307.7914440437, 428.3165579405, 252.6325482303,
    482.5793703218, 100.6855582836, 207.4137953397, 407.3575940149,
    468.5702351823, 588.8198027905, 531.3937365422, 131.3452137392,
    337.5826651754, 587.8247253992, 141.8864393869, 500.3923373054,
    400.3824396347, 140.0292419999, 283.4309146650, 667.8284309031,
    136.2600610558, 651.6271618361, 88.8318022357,  705.2303488064,
    159.8192021109, 683.2739937035, 215.2590065046, 640.2248726813,
    454.6189018461, 420.5719657854, 256.5664877796, 450.6886607958,
    287.5182407384, 682.0706665826, 217.7748582067, 39.0462660741,
    218.0137796065, 690.3888402986, 399.5977155380, 522.2067408689,
    405.3929532686, 194.9571483695, 467.8223289389, 715.9477754993,
    503.9679170843, 555.4753218466, 232.5318901928, 648.2472989182,
    428.6519404099, 45.9176844812,  136.2083425854, 293.1534368051,
    502.2837156053, 77.6927673004,  106.6850139442, 533.7446860820,
    254.4722719900, 417.3978184098, 381.6664451305, 704.4862018271,
    62.2545296088,  824.3719233222, 193.4864805574, 329.2407371221,
    163.5322776383, 796.7210636123, 171.6914849939, 256.5245967811,
    206.5842943697, 190.4650117966, 188.2565124745, 236.8750691960,
    420.4499316864, 144.5791427940, 639.3404979349, 385.3774752320,
    289.6684408534, 300.7427922815, 839.4766391814, 367.2516463203,
    68.3391568038,  99.6397056812,  309.4246710336, 537.2201564735,
    266.8936390506, 521.8190024603, 453.7911268652, 458.0707963476,
    135.5849998035, 178.7634931151, 259.5244315697, 34.2095838459,
    44.7987144504,  485.7495653064, 397.9448422609, 620.1809336586,
    498.2755076242, 133.5842536272, 244.5420554029, 210.5000000000,
    292.5000000000, 862.7752443774, 503.3170338345, 254.5400063431,
    548.0580175711, 282.3785158570, 78.9363752101,  274.0648191677,
    361.5054580825, 53.0979410000,  168.3850551026, 124.3500138723,
    304.7403704201, 584.2848998438, 263.4182004292, 287.4060904930,
    473.9934867930, 747.5017294939, 494.7767506123, 758.0980992971,
    560.8375556194, 499.7643463347, 246.6144125194, 194.6530032918,
    462.7918545647, 590.3917824714, 76.1490604474,  761.4594941092,
    229.9590116142, 237.7216925434, 462.5416184124, 791.6091001192,
    341.6526669076, 694.7741153341, 342.4983112695, 874.7225037844,
    142.6932143650, 680.7617714576, 291.8889080467, 337.7761963087,
    173.5227898154, 753.4118264946, 347.5892316583, 37.1323302305,
    273.8842276086, 777.5700758822, 518.1281889964, 673.7333639160,
    420.1741423182, 782.0488501019, 558.7848647662, 530.2529043146,
    252.9386017490, 672.0937479287, 285.9481572407, 835.2416004811,
    275.4505364534, 717.8100379298, 469.9631374820, 255.7848202930,
    145.3416661405, 850.9668680853, 94.1424719797,  254.4382481370,
    545.9119241282, 218.0397349072, 461.7377315965, 493.5858479262,
    171.1829693461, 143.4035159950, 195.2955625324, 306.0363645425,
    527.8843452805, 264.0526432374, 225.0262634147, 614.7373051793,
    140.1848355582, 511.2186393811, 513.2622498121, 339.6202149649,
    691.4260162803, 467.3378165290, 737.8219655995, 511.9396363182,
    400.2261887241, 854.1265612233, 504.8166034375, 262.9535618309,
    368.8766143811, 854.7044628921, 262.5679887032, 618.6465757897,
    262.9288832251, 777.7371988437, 568.5895172149, 174.4038038022,
    322.5258367003, 43.3879958970,  525.2912657241, 205.2986145212,
    471.7944228973, 28.4864594264,  148.2680004014, 70.9671883417,
    286.1578590144, 848.9011540980, 172.7805372774, 739.0631967819,
    254.3833249540, 192.2193077486, 387.7539229651, 377.0736156979,
    468.2789335717, 212.9810710998, 175.2770963625, 191.4299492662,
    871.0973766117, 138.1930311023, 364.4465024501, 268.2995271577,
    50.0573317927,  85.0211963818,  709.2693182540, 318.8823647056,
    321.3891414538, 567.9909805701, 293.9262330922, 209.1693842534,
    388.3073296410, 276.2101740683, 129.5233779433, 231.1911141741,
    36.1288236386,  89.6986558110,  369.1544941697, 327.5464872991,
    694.9703922781, 571.8497945682, 116.9997643099, 359.5581217427,
    268.5540905282, 363.7561550590, 843.6559608929, 464.9217682606,
    706.5054704490, 218.5967421246, 616.8789115065, 509.7600616188,
    756.5735077613, 546.6671256922, 289.7920870280, 352.6618603814,
    256.9773574380, 32.3198602236,  67.8690647572,  173.4520458228,
    251.7249089952, 40.4063490479,  594.7155837988, 432.4352871055,
    690.0915213980, 569.5000000000, 524.1544658652, 99.7283618264,
    219.5023697487, 485.4921899664, 409.9171843010, 267.4082593047,
    468.1151653831, 116.5519069329, 786.2313222331, 208.1760202457,
    226.3384974514, 85.2516819250,  821.9152706716, 271.1511271268,
    99.2400871788,  160.1187061929, 673.8728006889, 528.3738603332,
    863.3655508756, 389.6525361521, 210.4946526249, 236.6573848722,
    287.4943900458, 478.0444318993, 690.7528394870, 268.7242220686,
    393.6968634645, 479.5384847316, 458.5275879153, 564.1632344203,
    312.3686878252, 566.2994731457, 540.4954651066, 367.5094235910,
    475.9528367239, 470.6413644948, 207.6207706791, 356.8466040436,
    343.8816713458, 284.1276873616, 438.2709379779, 135.2227234382,
    286.5286205499, 521.8363754746, 410.2154883310, 426.6126009068,
    196.5551699579, 160.6805929085, 130.8439878141, 842.5565067767,
    216.7822113738, 794.3837124605, 569.3424678821, 336.1298141590,
    179.3982047096, 465.1517246928, 457.3796671871, 401.1234959443,
    277.4389526747, 772.4376483746, 512.2008758501, 677.7574168765,
    262.4721382441, 728.7541737614, 496.2177922305, 593.3531537968,
    289.9357128571, 156.5889073426, 354.3561756940, 613.8436794870,
    365.7281833317, 311.3245121907, 180.1830827583, 708.9987672354,
    314.1925660067, 717.8754424625, 191.8786209926, 486.1984529918,
    313.0376050418, 507.1099697999, 352.2004124087, 682.9994749358,
    883.5000000000, 307.4665035438, 402.7246498011, 252.1727884551,
    201.9964021317, 352.6667080436, 503.7060975335, 133.3259296711,
    381.1705078209, 454.0079253219, 243.5229164288, 295.9468141297,
    522.9381820463, 468.0092614532, 315.7988060748, 519.1580244670,
    446.8296427816, 186.7458967200, 549.5105396416, 250.3640355736,
    362.6994617827, 110.3426640140, 810.6153527642, 388.1700447076,
    210.3128242438, 397.8234765337, 662.1656271164, 498.3018861448,
    515.3035400457, 415.9115392726, 207.7522414561, 115.0259911807,
    545.9738676699, 147.9242373528, 225.9106686001, 367.3280794708,
    186.8762162703, 81.5795323365,  258.3589765767, 765.8755776601,
    548.6569667969, 575.6547637085, 387.0642522961, 241.0527573657,
    488.3961429240, 612.7536354696, 167.5595899166, 726.8797539869,
    362.8982697343, 366.8281253310, 458.2081846738, 184.4297445035,
    570.6829415347, 511.1117090802, 415.0961157671, 79.0786428283,
    62.5987022676,  728.7552573685, 409.7322237339, 330.1573847349,
    447.6867603332, 690.8333836351, 431.6067659820, 461.7902870783,
    98.7703028951,  438.9602177470, 788.0190191680, 367.9005631974,
    219.6557567253, 350.8358816065, 611.0851104215, 187.7432799668,
    422.5181486429, 295.0444133733, 734.2135079282, 566.9354930587,
    597.8150541492, 471.7973378895, 178.5669050127, 516.3655411452,
    114.0201916903, 256.7351163300, 475.0269804957, 15.9220345287,
    855.1252311366, 371.0423872806, 843.6231739541, 315.0946227072,
    154.0490129717, 406.4608296453, 205.4032316465, 272.3001730444,
    582.2830940913, 321.2979670244, 123.8868090828, 466.3162106480,
    784.7537847714, 412.6532150046, 603.7291904494, 342.1567176252,
    350.8393002913, 431.9584962862, 574.0378810373, 400.5168007575,
    766.0987030318, 95.2596953735,  528.4785628807, 518.0928150913,
    688.5505286801, 452.7707744142, 525.6830313694, 195.3260897864,
    654.7126790660, 189.3564345662, 202.7833858379, 162.6396481576,
    283.2758016528, 459.3589198996, 796.4225497331, 111.9306275450,
    855.6433236088, 569.3886881454, 379.4976756277, 223.9758299780,
    643.5471365034, 475.9767134930, 152.0159053512, 300.2940150457,
    530.6271220744, 145.9132440772, 829.2393324711, 577.6466540231,
    737.8570105686, 397.3556483163, 399.3082291353, 145.9903281782,
    695.8723148653, 269.0676973940, 794.8246878235, 504.4767373779,
    823.9027902144, 33.1129793449,  106.1114250991, 644.3271739174,
    399.8622557185, 835.2883024657, 230.7827104657, 271.8432046810,
    183.1359546153, 405.0234361727, 262.9962924577, 743.9638403691,
    234.7159241609, 206.1149731500, 298.6392163131, 347.2768487679,
    465.0581530652, 301.2258394456, 387.3345340403, 792.9681911540,
    31.5222335622,  705.9833262148, 485.0164399441, 546.5870455869,
    79.7786416294,  156.7352766156, 122.8819110847, 180.0172556246,
    100.6590612908, 160.0635769155, 238.9659552802, 461.6806673060,
    423.7312877116, 422.0361318989, 156.4565121335, 292.3718813264,
    496.1575763630, 73.2750477168,  238.2577463900, 204.3475286185,
    789.8394384835, 277.0531168080, 86.7694115905,  481.3905974738,
    442.3778631057, 107.4294161372, 164.3873452004, 271.2929608816,
    598.9600285823, 169.9145236252, 300.8834530512, 59.9505203401,
    677.8611141117, 250.6133976711, 414.3517397671, 99.6412370083,
    280.8243937194, 396.8588793816, 116.4831446507, 283.2005206267,
    413.1890532706, 352.0075373283, 588.1567507478, 346.7968591542,
    171.0537739146, 297.3150393830, 627.1189430648, 161.0262533014,
    403.6271624373, 263.5929096777, 792.0677989841, 281.8524517075,
    171.4949906028, 90.7137287168,  653.3474963643, 86.9035764491,
    533.4506907631, 510.7062719042, 338.1751256562, 226.8047049356,
    320.8573242544, 318.9760575659, 783.9048189300, 415.0791907493,
    102.0715788589, 148.9586061654, 728.4715837811, 479.0901046147,
    477.3074938055, 337.2936144520, 681.5745404755, 124.4842998940,
    649.1077975027, 338.1546274125, 628.0957066934, 328.0620049883,
    78.1336855944,  132.2461366362, 721.0391229960, 321.0970804251,
    214.6768025237, 115.5620149203, 239.5568456449, 29.4005953937,
    617.1124408854, 488.6639750961, 264.8137175777, 223.8434411151,
    217.4008174993, 509.7268600883, 446.3885897390, 36.2968130759,
    256.9440449766, 147.6743838878, 131.7339062035, 200.8968257679,
    608.8711619247, 474.0859426513, 86.2278026839,  502.0188829816,
    306.0514988556, 453.5222792915, 583.7210453858, 480.8630202978,
    853.5872191107, 241.0319124627, 835.7507560770, 16.0142377840,
    65.9820483645,  393.7981386785, 804.7255034818, 567.9135808335,
    271.9276191800, 431.9240042273, 305.4355579799, 349.6993806196,
    318.2134308971, 577.0133648724, 344.0023856657, 142.9814354957,
    236.3610111198, 584.7133997682, 318.9543854822, 498.1032076625,
    130.8350539739, 437.3222433167, 473.6244130367, 880.7534924951,
    173.3144748988, 254.9129620672, 118.8027959959, 352.9246822862,
    396.5842569310, 415.4595461803, 170.4477758971, 693.5793207437,
    103.7661690469, 771.9352848048, 181.0642497450, 94.0131635726,
    325.6151051977, 190.3146479500, 18.9358605650,  607.3678996354,
    336.2950504142, 62.7847433589,  86.2955726397,  492.3905855669,
    164.6482762581, 242.5591549361, 87.1268649314,  701.1910670681,
    125.6335729449, 772.6878846047, 355.4895634121, 333.5668840897,
    562.2918480534, 207.1744050555, 73.6124394554,  435.1354334485,
    162.0273297741, 174.0715844129, 156.8382210070, 513.5303834188,
    188.3049739764, 72.9653274678,  414.5107077640, 75.5323587813,
    283.3754873063, 722.7639409791, 451.5807510887, 261.5572188451,
    185.9303060573, 859.1958787550, 345.5369863027, 181.1500902664,
    169.7486257404, 648.6493267968, 156.6828582407, 595.7828247249,
    126.7257921465, 561.3786238267, 456.7255906462, 787.4769628176,
    316.4717916024, 576.4954887114, 481.9516765202, 98.3852105308,
    106.6958446791, 128.4697231089, 415.5746797222, 491.8544632501,
    244.6621124884, 393.5563442187, 292.6430324604, 524.5735396489,
    303.6274209201, 248.3698244627, 340.7506056615, 359.6998869403,
    452.2304793817, 714.4791394360, 536.1939805491, 855.7400286777,
    384.5683644231, 148.8349768451, 275.0436014235, 271.5836040640,
    192.7838282995, 405.1654318004, 533.8833628471, 23.3741755182,
    685.3279760120, 445.1578480594, 819.6715237954, 75.4706799279,
    842.1356927879, 462.5810227394, 463.3954917359, 89.1797527902,
    103.5596584963, 300.2196626721, 497.6877487648, 471.8223906459,
    275.7389866135, 428.8810382767, 788.6937121280, 449.8869768867,
    425.8306414562, 69.1077412316,  113.7450707104, 186.0948785140,
    623.2285861214, 366.6074253820, 611.5712530825, 131.1815976730,
    791.5768088041, 294.1364744979, 319.0274615942, 137.0737469103,
    738.5117671213, 292.7950225960, 851.7377034731, 184.4367269742,
    827.6791673684, 508.6797675759, 487.2673155814, 241.0827156115,
    91.7702682068,  377.0564448291, 865.0706515513, 463.6753131137,
    119.6486625631, 562.8566835627, 745.9217690354, 255.7003110682,
    131.1657108043, 232.7006190461, 354.6950222506, 574.5000000000,
    632.1491477362, 521.1436929601, 382.8243129962, 341.8374733072,
    240.6895366531, 249.5232551785, 498.7326333800, 451.3195565247,
    504.1926489476, 507.2433581105, 373.3474018846, 354.0111108066,
    183.3174326692, 449.3793051641, 222.3730232202, 172.5697177124,
    663.3902179576, 565.1119021759, 290.2210361178, 494.1515501767,
    354.2432551599, 225.3282196811, 376.1381774230, 416.4367637754,
    484.8711418033, 244.8043972161, 370.0948930112, 390.5394732947,
    860.4591347643, 509.6543083551, 99.4088531227,  297.4524216205,
    480.2172527878, 459.5124211693, 635.9605141371, 341.2882017997,
    598.2865279087, 478.5792901954, 631.7512778931, 57.0456718615,
    675.7512592586, 357.4073051822, 619.1392863800, 558.1232315106,
    44.5604506824,  18.6170476304,  573.2349973447, 264.5690128899,
    263.3709831900, 407.3232645569, 198.2954827585, 460.8600193450,
    221.9092048252, 309.1631456608, 276.8306060741, 466.5677340383,
    701.0243920902, 256.6905686123, 558.5256358887, 325.5375967570,
    417.6498822274, 189.2853096387, 151.0522779992, 37.5319905031,
    502.6456016026, 450.0090436676, 43.0355671315,  344.9825352878,
    479.7423126905, 279.1487531746, 697.3759237683, 94.6749258549,
    347.4984444655, 233.1269132477, 436.9890878920, 471.1372573852,
    668.4979558937, 445.7891710020, 716.3860146454, 73.3773269427,
    358.5678369783, 255.1757369973, 433.6632150040, 183.9160248621,
    178.6681088478, 410.6899738731, 592.7190955499, 281.2532707342,
    445.7289609102, 260.4412642042, 101.6539006852, 28.4766843788,
    467.7894737248, 427.6108606825, 70.1620609522,  514.6641879247,
    266.2187253232, 594.4049810482, 154.1867783030, 721.7397761869,
    449.6797869184, 738.7187680109, 106.4121817265, 324.5565347026,
    523.4025554365, 248.2706134183, 324.1129666971, 63.5958401275,
    539.4046284602, 451.9338750698, 368.6715013241, 581.3084315401,
    399.2252521727, 81.0872475128,  125.2380676561, 211.5000000000,
    286.5000000000, 199.8000000000, 617.1345240048, 398.7470344886,
    816.6566005273, 426.4620777677, 81.7730059360,  540.3510983785,
    230.3255602236, 390.9088217108, 119.0843418558, 508.0553256868,
    590.1622881156, 575.2697050364, 764.0784346893, 273.3560447025,
    54.0053571655,  548.7097261571, 27.1464837007,  124.7022750840,
    798.2907011588, 161.7439768037, 500.6506923992, 324.0666999976,
    777.6925747468, 173.9867669276, 638.2929223595, 297.1079412667,
    391.5089986157, 365.3517510920, 724.7902793611, 466.2979329074,
    220.1349765090, 111.2532443083, 371.9139550393, 413.0123818425,
    405.9516422014, 335.5379122198, 441.9032353935, 235.6508779687,
    258.6107615216, 225.8433674684, 427.5058819938, 149.1947740122,
    518.7161075246, 356.8129449187, 689.7947209709, 382.2697419271,
    749.9207006495, 432.9427378684, 475.6347139448, 375.7253169158,
    767.3298468392, 482.9307523328, 405.2837229566, 246.2996342638,
    481.1576192966, 488.7145265121, 308.3553896594, 390.9148400883,
    608.0689320032, 534.2692418426, 637.4550249259, 469.6628722431,
    480.2400521559, 390.1622014838, 723.1782101568, 821.8691153992,
    565.7902341478, 280.1099992622, 455.0816599007, 713.9366820431,
    514.2638436133, 468.9775564592, 249.8152260218, 568.6020935759,
    363.6102929840, 600.9733456407, 36.3333223028,  412.3749006933,
    541.3359807271, 559.1681372604, 237.0305790769, 642.0592900481,
    103.5086677276, 166.5248613748, 500.4085911325, 680.5072765542,
    331.3671316612, 353.5142561657, 196.5979766942, 745.2230163799,
    579.1686524131, 299.7810709040, 572.2637190203, 105.1305464250,
    547.5428497583, 440.4240992946, 96.3351560476,  329.2525228890,
    443.5293168834, 277.6774474055, 170.4522118287, 398.8490739766,
    314.5283504396, 696.4486491772, 352.7735765342, 484.1554384273,
    181.1305533033, 535.3067965676, 362.7590371444, 466.5027996042,
    375.9256162151, 294.9245917401, 116.9826253714, 275.7516448441,
    487.9683271461, 587.1071339153, 168.6939718363, 592.5140477638,
    470.3620754360, 645.9817343733, 303.4203161733, 471.9007680660,
    88.0180427593,  717.0018954270, 197.7700918055, 187.1765337052,
    352.9661219642, 379.4075446081, 116.0330910326, 88.1299487699,
    448.0318870231, 616.3663592460, 178.6581819847, 408.6012397757,
    64.1066498653,  273.4842523538, 546.8743571346, 182.1255452755,
    337.2196356431, 330.9481130211, 217.7072835209, 58.6205147667,
    217.5765654719, 234.3650152382, 139.1616368622, 711.3591303221,
    223.1772024093, 617.8434926936, 521.3114202029, 111.8417003645,
    26.0401544026,  533.5068715122, 138.0938002922, 854.3010033671,
    512.8487594599, 702.4077058194, 413.2787142184, 111.9795351790,
    510.4897615346, 693.9546592149, 399.5936924686, 446.9625524573,
    224.2022278066, 390.9749517372, 221.4081692155, 588.1693657874,
    255.8528824560, 577.2685457817, 555.9943927087, 843.4297029468,
    144.7309432567, 495.5155437215, 380.2405180398, 316.4933372033,
    43.2355659653,  322.1193449007, 484.6982041905, 762.8293378319,
    436.4157401314, 820.1565904443, 241.1122399686, 44.3253512905,
    151.0054741300, 535.4099188455, 375.2561701674, 70.0932516325,
    22.4945177050,  756.0819272375, 431.4131141517, 345.5659978806,
    114.1310633857, 537.3910625057, 68.8699456026,  244.1316675011,
    436.9886854833, 782.5528184277, 238.7464441257, 149.9871312866,
    452.3464640662, 707.3389312016, 424.0107455285, 329.0428834720,
    357.3548931927, 840.4534026430, 500.9222569192, 247.7282284196,
    307.0425526440, 784.4468128957, 367.1359281482, 334.3618577969,
    95.0300966135,  216.6028938665, 364.1888751982, 533.5888822885,
    538.9575355661, 230.7740894302, 274.0711933107, 821.5272717879,
    396.4063831836, 564.0669940378, 578.3641142724, 212.5000000000,
    273.5000000000, 820.0122886408, 555.6441957642, 544.6095127602,
    367.9171191072, 733.4979407705, 497.6453112891, 863.5777134207,
    140.6335870532, 122.5864976638, 228.2182590682, 219.2866803697,
    468.0218121968, 78.0860242602,  224.9203496790, 868.5706573787,
    458.6096365585, 404.1962428607, 20.9601048299,  755.6544303219,
    319.3065169835, 498.0275750978, 110.7229155098, 124.3796084156,
    833.9526495150, 405.9198141142, 42.3816242309,  447.5194281556,
    733.9780528636, 65.7819571538,  549.1365815381, 251.7842174555,
    483.9250948131, 424.1581441105, 282.6202036456, 565.5220643584,
    291.9099133332, 401.9494943701, 43.4324938265,  732.9596093199,
    224.7270341756, 861.0311828169, 566.1714866784, 517.3143874751,
    235.1543815522, 449.8916663542, 334.2308732691, 241.8435817212,
    236.8998313258, 85.3274896510,  326.3908471987, 538.4608238521,
    377.0585843765, 506.3569303093, 254.0365051101, 450.6165250379,
    313.5175257344, 129.2076914051, 448.8140960936, 832.7234256223,
    248.6701133515, 539.4725597540, 63.7030533210,  197.2716401810,
    35.6674016419,  210.1133999827, 555.4045969667, 558.5485089541,
    472.6772249609, 467.8482775472, 299.7377586393, 862.7443177454,
    79.8512193246,  774.3161094769, 118.8607165825, 68.3732659053,
    403.2431159762, 773.3756953393, 147.0714116586, 43.1045140676,
    573.1696317320, 731.9232767438, 265.0362378182, 407.1644241441,
    194.9247525665, 132.4382508724, 323.0820752663, 733.7974753387,
    512.8359665480, 591.5025046873, 409.4926464484, 553.4780624292,
    241.9875152814, 711.6112498269, 478.3878084228, 656.3634248594,
    462.1695846649, 696.0853249341, 497.0467208213, 554.7031003443,
    304.9246163174, 55.9464976204,  551.3082876723, 56.4664004567,
    328.1924288460, 130.7025244276, 547.3318959153, 650.3410002831,
    309.8006127593, 861.6416591564, 512.0162476207, 124.6500056126,
    198.1825928890, 665.0717430877, 304.8915263086, 735.8899504389,
    106.0823698256, 812.2910019411, 110.7982996666, 463.4787724542,
    309.9724209044, 553.6546283933, 132.9408170282, 96.7427047632,
    425.6288735435, 252.3577097322, 153.6808765436, 784.9745774513,
    192.4232796647, 752.5786316723, 475.4278806700, 648.9595777480,
    180.9303327890, 191.0058320470, 302.1505831014, 74.5647233443,
    91.4888158492,  366.0549904063, 390.7990577672, 593.1326095856,
    28.5231303220,  748.3165691962, 569.2256750820, 101.6806720557,
    249.1967564920, 152.1106740210, 68.7467502549,  283.8971522497,
    582.5000000000, 303.8995041955, 106.4620089364, 632.7337006300,
    191.0337664066, 137.4560796890, 522.6319190513, 202.9777711829,
    177.7567476824, 790.1770104507, 269.3972528234, 471.7426143450,
    260.4208609575, 718.5311401667, 448.4462208776, 623.6338043186,
    493.9571869715, 666.9041998929, 276.6538101544, 810.7676975034,
    534.4293873087, 511.2109221469, 45.6439802064,  300.8462525029,
    268.4733035993, 802.1163360305, 341.1041517586, 431.3360358407,
    190.5287868457, 378.8977419205, 538.6010498317, 834.5826726426,
    403.5847961197, 442.3115047168, 256.6090144763, 605.6859151090,
    502.2374621196, 325.1284461894, 490.0636660628, 16.5616244561,
    258.8580094289, 166.7016039937, 403.4872588976, 832.9022671071,
    488.2308664746, 317.1605945904, 219.2012808291, 137.7677915224,
    374.0709933583, 793.9652734656, 385.3933696637, 119.4308309853,
    465.8102467744, 155.4171889478, 47.3819871075,  340.0441502992,
    273.1819437793, 399.1326663173, 369.2615209586, 241.6866288990,
    116.4237353218, 415.7258805321, 645.8358232090, 363.0567234473,
    616.4154709448, 47.2631595834,  715.2576638377, 361.2738056875,
    667.3659825598, 364.1596613149, 136.2090333856, 501.8444879129,
    474.2117502075, 85.6465137404,  496.5722219167, 423.7165434603,
    631.4146508718, 257.0359324680, 568.3283992839, 430.1890487553,
    849.0570672061, 336.9809920106, 616.4930023936, 308.4555988029,
    94.6727563054,  197.3500077200, 685.8162722829, 266.3218838992,
    790.5071925146, 211.8405166628, 153.9353945475, 198.2987850168,
    105.8400840138, 462.6211968464, 72.6948719059,  548.6399432755,
    350.8140519788, 296.4936184049, 412.7432486520, 58.5524094825,
    679.8730216753, 273.1860239338, 792.9334776230, 284.9730967782,
    240.2619127888, 170.6848350989, 739.9034195220, 365.2006365440,
    517.3401691139, 479.6784149287, 159.6680764644, 453.2585038032,
    563.5840623868, 331.3817796395, 698.7596466375, 474.4226649795,
    802.6948532213, 423.0905851001, 728.9704573600, 465.2963674908,
    178.2330616554, 126.2695801971, 502.0058474182, 426.4710232866,
    255.9294567300, 338.2385348779, 463.5051086898, 364.4270978572,
    278.7846838824, 74.9631356998,  275.3495311776, 164.4888501161,
    120.3797405583, 120.3758860356, 411.2007643865, 485.4601057221,
    174.1518964709, 452.7277549246, 476.0381678037, 75.0584592209,
    279.7349354546, 312.4578845292, 510.5893281052, 204.0214100872,
    563.4097461538, 385.8370690586, 624.6749222238, 295.4713256612,
    663.7880425547, 51.5816679115,  569.9398904689, 262.1684312840,
    664.2597825044, 23.6330945481,  843.3271913581, 219.2823974596,
    198.0103968065, 530.0209164462, 352.2188302359, 459.6559220224,
    804.4356343060, 305.0557069800, 266.0826555510, 123.2205074375,
    184.3680600332, 554.6558995900, 570.1684695561, 281.7190801233,
    265.8167667751, 72.5323325674,  467.3523934680, 299.3436529355,
    455.1864314794, 32.9249257685,  518.9660948062, 421.8482832955,
    386.5796087808, 564.4169280481, 118.9128387110, 317.0567633211,
    522.9375822318, 274.5201104622, 170.5958563611, 305.1473507801,
    493.2088258255, 555.0617400394, 440.9890293994, 286.9802675196,
    220.1566403481, 139.1801439648, 872.8817135451, 326.4084712280,
    626.7888452281, 310.5599335124, 672.5662248681, 176.0204107885,
    239.6338921283, 463.9961003601, 752.5372792141, 554.7794580535,
    549.5263118178, 494.1900262260, 553.1578313827, 482.4209129022,
    71.2752382695,  298.5904609802, 848.5336379824, 158.0406534268,
    96.8968333706,  451.4128294532, 367.9871499940, 139.2989965877,
    820.9057445254, 241.8758180630, 698.1282618086, 148.5866664082,
    370.3715146267, 28.5609838542,  679.2115523990, 231.6128235294,
    378.0460553063, 547.8262887434, 679.3052063661, 217.2083623843,
    148.0030292990, 435.3883592755, 206.8864260212, 132.4902518323,
    487.4480917049, 247.9192607195, 176.4678789080, 463.7572033540,
    141.0355333844, 489.6539810522, 641.9832696858, 288.0710916449,
    335.3422537871, 78.8636788882,  528.1270993334, 574.0186671587,
    416.5973977800, 511.3758525297, 804.6863258241, 580.0620838780,
    521.0440793815, 377.6676862790, 676.6170834492, 473.5745359166,
    620.6241985837, 231.1849983211, 685.9271365622, 556.0851565518,
    859.6657376327, 358.8167365801, 136.3586794804, 163.8969073261,
    733.7681814531, 53.8773492345,  678.1702253311, 449.5554703206,
    670.6167476759, 146.8945721839, 397.5690796564, 634.8460860811,
    206.5207845028, 77.1924728309,  342.6994827842, 275.6117647958,
    379.9530372841, 668.4370109985, 106.5991597995, 153.4439933831,
    192.7305745511, 877.0548658269, 435.9426104523, 24.5000000000,
    501.6220720342, 451.0916395189, 204.9122725365, 675.8959990659,
    398.9252014618, 397.9139233175, 415.3026437605, 321.4488022608,
    497.5559144013, 451.1625454729, 30.8416731100,  262.7639600646,
    396.4590008739, 695.1985550973, 485.5548357416, 590.0004591485,
    367.6432675867, 25.5000000000,  121.8931692009, 399.1714358643,
    551.4524474367, 58.0282540324,  285.8393116341, 161.9047656748,
    813.0377926386, 574.4507409234, 564.3339957197, 57.9355725949,
    416.9388029036, 154.3584194770, 157.7953679478, 406.3841191719,
    253.4391657052, 328.4346192438, 288.7654682945, 419.6682533698,
    99.2819336678,  560.9980791282, 834.1221373133, 102.4432571150,
    432.9320402355, 392.5558269246, 726.1716208182, 393.4953941008,
    166.0364645213, 96.5590757181,  508.0901288703, 473.4311831216,
    596.7313484868, 525.7741072382, 472.6074271817, 52.4470997604,
    92.8122861625,  352.8121598620, 737.6950240014, 261.4202996522,
    162.9354943663, 548.0228479964, 118.8157814339, 239.0240404960,
    701.4547611806, 281.0496425970, 20.2036613828,  478.4218473664,
    40.8175813658,  234.2887555326, 295.4102360799, 345.0520543952,
    66.1373049179,  367.2430249964, 101.8281250780, 68.6939198898,
    264.1018304231, 488.3919908541, 400.3765470046, 41.6361996841,
    720.9329661743, 351.8349205587, 695.8180724599, 344.2839180875,
    742.0074324399, 144.2949753193, 355.8627098786, 230.2807597251,
    190.2278269455, 560.1908955403, 233.8885842781, 393.1100523023,
    324.0251160008, 380.4823948328, 619.9501893282, 422.4802500630,
    299.0421732770, 124.9216808573, 511.3420047711, 454.1305288391,
    89.4746433224,  144.1039666230, 277.1124231808, 262.7666754971,
    803.1732327787, 448.8140239406, 583.2639892430, 198.5905077460,
    83.1114666410,  370.3084085660, 872.2085512676, 476.4340749267,
    484.9116106220, 360.8573016915, 772.6057426368, 282.5237814987,
    507.0042091214, 287.1073533868, 567.4828967717, 77.6638236072,
    691.1121680399, 456.7582312892, 121.3765148875, 132.1427164631,
    263.7887606281, 467.5290794598, 713.2332481745, 209.0405124103,
    512.0876058071, 438.5789167409, 246.6029602691, 384.2014472740,
    721.4169417444, 72.9779419625,  608.6473249425, 328.0311011960,
    313.4962762107, 459.4197606034, 542.0705413935, 322.8168425219,
    629.2394460797, 23.6288793215,  653.5719153655, 360.4089140851,
    582.2139991666, 163.8328830829, 621.3713390844, 497.3343778797,
    871.8811445324, 238.6704807058, 204.0570714829, 826.7269314280,
    255.1804404453, 523.9480584399, 449.5299969837, 335.4295995321,
    149.7863918270, 454.7429014134, 102.3244946742, 502.6301853078,
    180.3489733917, 332.6747864283, 259.6430604050, 230.8772845444,
    560.4429621060, 296.4176472799, 219.0217740151, 116.3082515091,
    257.5868649798, 554.9964123956, 453.6092829338, 172.3277598091,
    437.1656535443, 33.2912818699,  519.6016003523, 269.8884015175,
    348.3783328032, 338.8614039587, 764.8175087663, 254.1132427399,
    849.6462157379, 481.8001836843, 778.1360778019, 245.7504044664,
    68.5839537093,  195.2958974697, 768.7059655018, 71.9757617704,
    774.3291194745, 130.3677109490, 367.4284956499, 461.2664125059,
    162.0689102483, 230.0654948111, 371.5185411738, 235.1079770159,
    182.9111909426, 415.8328965689, 279.3676699127, 48.4862395811,
    112.2859858350, 85.4527716332,  190.3006181641, 466.8324650799,
    296.9772299695, 436.7872589982, 649.3853741424, 160.6637209782,
    726.1933457741, 185.6285572668, 179.5126695824, 754.4497487399,
    166.8606597707, 711.1544132514, 513.4264797779, 716.7615520789,
    325.4252504193, 802.8131853606, 61.7555448309,  299.3589952048,
    202.5493178688, 362.5807019437, 252.7866297237, 178.5983602950,
    351.7999857042, 232.6682340856, 408.2613972378, 820.5174221858,
    151.1805944706, 442.3805853850, 278.8521684476, 587.8779826082,
    558.2107509045, 610.5912241546, 214.7978010602, 880.5000000000,
    407.3620010627, 424.3382600225, 504.4861445521, 571.8620738460,
    282.6388077858, 298.4593183289, 223.5049699746, 427.7934864283,
    453.9395727332, 786.9816148269, 406.9181668460, 465.2963000458,
    441.6367461912, 597.2897407766, 450.6501420868, 794.0888075935,
    485.3491176714, 646.1344765308, 307.3999172242, 522.9371011051,
    439.0671352442, 596.4051597772, 271.9569527638, 704.8304456469,
    116.4877670719, 679.0757769097, 208.8706470673, 478.4900202739,
    567.6677374937, 244.9922597238, 176.3144633075, 398.3576770003,
    274.3748793873, 156.4203561054, 169.5461561150, 195.6236713418,
    437.3724799878, 68.7288123631,  222.8184819547, 260.9504574946,
    221.4449380550, 747.2548169790, 551.0003540676, 79.3797054021,
    188.4105439389, 525.6107520488, 373.8820214634, 230.5689308501,
    562.4786591538, 719.1384150168, 241.5176103456, 150.7132025240,
    32.1273031231,  697.5549188394, 310.7529082863, 315.9274164218,
    267.0188643646, 206.4161182735, 166.5483236937, 140.1703483119,
    533.4942556566, 610.9678897418, 415.1746886915, 847.5281064699,
    561.9031764623, 385.4993156304, 557.6038920327, 621.4236088643,
    489.3500352106, 662.7090995610, 310.3478538342, 543.1513947513,
    53.3405795089,  470.1885162133, 537.0309770600, 845.8579155872,
    150.6076098027, 598.9326206872, 426.7003687590, 844.4283095903,
    578.0585406062, 569.6676116209, 406.4738281011, 584.5066893307,
    280.7221352719, 777.4371794192, 362.8772955575, 769.9491265178,
    226.4691002450, 384.3182527322, 269.3935827160, 291.8315730741,
    438.3627363506, 504.7212415182, 84.8146096720,  289.0463800053,
    169.1661213981, 513.8783466274, 31.5868844984,  480.0234430000,
    252.6084815577, 128.2478175486, 527.7429819700, 230.4971954513,
    458.7293877910, 448.7501021552, 447.0734982846, 714.1541582896,
    559.0492298435, 526.9326737146, 442.1734023097, 449.9457136017,
    205.6449656386, 357.9133121917, 361.9923214523, 273.8675591290,
    131.8544719048, 720.0681714723, 98.5333374848,  707.6394793132,
    115.0750208184, 211.2449241526, 485.1515593592, 387.8769971161,
    77.3226980078,  696.3914721127, 522.4537022390, 759.6986003192,
    315.7546196269, 134.8750148341, 195.9596258054, 149.4126448810,
    547.1512838903, 44.8077658242,  169.4172520820, 422.6260278045,
    564.4525421854, 180.5627179055, 340.3380289137, 878.5994563229,
    410.8579710060, 28.1807605279,  521.9307046183, 347.6646177391,
    549.2969670442, 213.6811320029, 175.7525603627, 503.1614426497,
    289.3958670593, 559.7726822634, 245.7456637139, 669.4705086475,
    546.4973907248, 377.8501055922, 483.9154513833, 557.6708731062,
    260.2124729198, 724.9855566955, 120.9375490930, 194.3462122359,
    347.0916845236, 757.9861160159, 434.1933606188, 422.4407069819,
    178.8088591436, 318.7096392550, 562.1119046258, 591.3938802173,
    521.5541188908, 771.5465580195, 73.4220522701,  343.1665570209,
    91.5394245617,  441.0202341279, 161.9707982915, 592.3953172522,
    142.5788297051, 66.1476981022,  37.7847340182,  842.9089913289,
    384.7781932026, 510.7847191763, 521.4284183935, 748.0048359080,
    254.3234004735, 394.6246904320, 188.2032436761, 850.5507490889,
    518.5071111191, 307.5369830419, 495.1689335475, 856.7662521290,
    361.7922064361, 351.9881588743, 364.6879206350, 18.5000000000,
    284.8621338164, 379.3257641680, 511.5744214132, 315.2766621377,
    96.8721562430,  631.2576917132, 418.8692970904, 229.6010010539,
    297.6511615586, 213.5000000000, 310.5000000000, 219.8192913005,
    829.8679219845, 319.0204437574, 283.6043698835, 336.2804998924,
    879.2507655801, 529.8661845092, 441.3812785854, 342.0056963690,
    416.3384018026, 564.9176945171, 17.0523443546,  256.7579219868,
    616.8194226233, 497.0303294030, 551.6397150881, 40.8397690965,
    794.7046375849, 385.4151014198, 677.5994220622, 192.3252471994,
    296.6336881878, 246.9982858891, 212.4154712524, 181.9230000300,
    718.7420763636, 462.1795420080, 750.3990285575, 404.0127711366,
    444.5237202150, 496.7932608167, 603.4406944826, 402.6714288921,
    414.4628263685, 320.0082332440, 640.1466027347, 297.0574674004,
    702.5912420635, 389.0127441152, 768.2868333738, 123.1951411596,
    374.9437158722, 486.3861117598, 44.7580069713,  357.1053550234,
    822.1069283867, 348.6579641508, 552.1221039741, 371.2073480687,
    408.4737592379, 109.1873134054, 430.1180898159, 333.9661151048,
    264.7220697013, 423.3111369182, 446.7984650890, 301.9076616378,
    816.2163497698, 245.9560277175, 404.8804266839, 374.9654249335,
    526.4642292975, 535.2303192588, 510.0722775435, 323.6336032238,
    253.3729940081, 436.1105475384, 433.6089531661, 404.8002473087,
    271.3825959154, 455.7133666318, 321.7995266819, 459.1267308515,
    179.8802368948, 94.6784300716,  372.1960287151, 569.0751497886,
    241.8947157229, 126.9567354394, 594.0259343061, 327.9182244169,
    583.5948754673, 373.2619511911, 513.1057775771, 68.7346142823,
    610.9349253144, 333.4269710134, 647.9065302285, 534.1577378502,
    141.0643783326, 407.4254584021, 855.3510727711, 279.7737839324,
    598.2586831597, 85.7992260046,  781.2640444420, 352.7639318957,
    243.1718153278, 292.2133949181, 836.6131401916, 133.1416785951,
    644.8878495363, 287.5637983082, 675.6127287214, 309.7105877587,
    176.8293660523, 224.7238416567, 578.2828430248, 531.7846927899,
    311.3692643718, 387.0643126167, 788.2080577208, 563.2561894045,
    765.4155935505, 129.2067174785, 460.8340228593, 424.8139457242,
    176.8742336457, 383.9455640422, 634.9894813039, 572.1060772182,
    160.6356244616, 436.5253575350, 95.8308132361,  90.5814022304,
    232.1697849059, 485.1747605468, 697.7059777731, 327.1194614442,
    370.0169975264, 567.0292960128, 719.2591473022, 499.6266974463,
    641.6680864807, 269.3278178414, 604.5986430529, 125.6702708434,
    352.7242636428, 382.5140900795, 647.8618031754, 279.0539325605,
    634.8861203698, 400.0217457803, 115.0518954726, 414.0739686365,
    653.9408926088, 65.4721766181,  828.7682231072, 224.7976491950,
    164.4039144643, 457.8286974731, 660.6199094824, 460.5918126618,
    858.7326879641, 160.2739107841, 831.7857756192, 437.2727526565,
    557.4618095516, 336.4779827311, 221.5153373008, 363.5379913736,
    183.1480998833, 445.2882523887, 286.3505780145, 245.8494462972,
    378.6519558128, 543.3393395211, 319.1672117713, 106.8274715755,
    531.8519313069, 332.9228224957, 199.0145146414, 262.2811728528,
    466.2729028214, 618.3849432178, 561.4070490292, 237.6409371623,
    267.5312433816, 233.2993495004, 368.0862767341, 580.7995401801,
    483.8229744608, 669.7469936788, 268.0951516727, 835.3332562313,
    353.9319947761, 448.6291595480, 86.0106151162,  360.0569631053,
    484.3813319895, 761.1529135012, 530.9333820874, 683.7260092305,
    454.8580778705, 814.9194069509, 486.5977870945, 94.6865546229,
    318.5775683960, 124.3285159886, 388.4165851435, 648.9205602789,
    130.1682438100, 278.2651922385, 118.0866762952, 82.5988208700,
    263.4975108346, 277.1914025654, 306.4621987841, 452.0447216317,
    565.4839299172, 427.5720574702, 378.2298464275, 98.5897547299,
    451.7487111435, 357.0478190024, 434.3880396244, 18.0897036234,
    560.9467083696, 504.5615039879, 235.7270650065, 178.5736328995,
    865.7728303724, 199.4691903023, 504.3603146142, 178.2560057852,
    323.0695956460, 487.8186837672, 691.9835864077, 352.2030700755,
    242.1982782113, 530.7352619668, 243.1200070816, 482.9276494686,
    270.1682580181, 101.1920769771, 529.3450506807, 451.8198467988,
    567.2773639348, 412.9646467172, 113.3044189439, 61.6239719360,
    19.1814321231,  160.6861796441, 224.4432294818, 132.6211320776,
    563.1526755407, 373.4178588396, 722.1069520230, 472.0747703094,
    869.9741209536, 335.6690180733, 215.8055158962, 226.8494509352,
    520.0947921137, 340.9958926679, 401.8708503244, 506.2681259310,
    380.9185544124, 93.8695440132,  619.9938435830, 300.2598889143,
    73.4496746840,  755.9005052153, 478.5176270985, 389.0028055071,
    480.4503848303, 728.4615452025, 76.5972729853,  386.1860827966,
    458.7013400464, 558.9650097048, 418.8396962155, 468.0048012483,
    444.7963432104, 721.5116282243, 419.6537896420, 614.0495625983,
    333.0274204198, 321.4449434791, 64.6965337603,  256.3318098322,
    390.7162246578, 388.8666931108, 302.4619598750, 792.8384088608,
    378.8583179243, 586.4166082123, 462.5337990518, 472.1744771545,
    271.0364972190, 580.5029075631, 342.5525664712, 717.1356684896,
    520.0615909669, 783.7389093081, 319.5557161677, 87.3707607263,
    527.3019215117, 83.5394074991,  130.9424463231, 787.5984417501,
    322.3847299648, 530.3221724212, 25.4459814518,  823.5926392050,
    409.8333800241, 214.5000000000, 279.5000000000, 358.7528776265,
    388.7208251683, 249.4155882029, 566.5000000000, 116.1057216746,
    188.5149031870, 232.8430013640, 449.0264370512, 274.3594464981,
    485.9560968261, 215.5095523995, 494.9943624393, 739.8479898735,
    336.7773839412, 476.3681730020, 176.0296313486, 256.6838187028,
    348.1229250706, 198.0666195662, 225.7909368261, 540.0316656605,
    553.2028052570, 220.9857551897, 424.5784348066, 47.6498367151,
    285.2312708240, 290.7119995935, 339.0030043263, 868.1064323252,
    503.5545522068, 681.3130518774, 421.3803077462, 565.8837894874,
    498.0496664484, 840.7491286861, 356.9871255465, 622.2089029465,
    142.3444002694, 437.1102872897, 197.2545073019, 43.7520564229,
    364.0923112161, 366.7483653682, 460.6192254920, 805.1323664987,
    425.8514923567, 677.1976240404, 283.8235922772, 518.2831833527,
    423.3123394140, 635.5201567753, 117.7158672888, 217.4629803272,
    461.4853970574, 426.6192228800, 226.5385378286, 562.6231974689,
    137.1113468128, 109.7095852140, 652.7399764249, 125.9602264292,
    182.6772600497, 95.3666357239,  326.6147341515, 173.9147147181,
    789.1249341787, 567.1161752205, 784.7111941190, 409.3437928284,
    522.8689743010, 337.4548631909, 461.6893176408, 78.6783348838,
    273.6633931358, 164.7204461124, 452.1618082564, 335.7813960725,
    263.3402679425, 345.0466393462, 364.2999156028, 388.7877328651,
    579.9134483355, 73.5940142001,  57.1567595976,  689.6136570600,
    267.5663478495, 679.4321238051, 180.0124723676, 532.0821706496,
    465.3902242265, 171.0914485359, 358.8634117482, 483.7389017240,
    490.7345272295, 477.2577879147, 463.9150216846, 528.3002005336,
    475.7477497260, 235.9645423258, 370.9086380255, 624.3686639412,
    190.4190817821, 37.1538215588,  163.7875441968, 281.7841730888,
    438.2780576391, 612.2018979684, 485.1061600847, 57.8798459013,
    203.8961308219, 72.0171112333,  525.4281834727, 603.4193530731,
    48.4597356202,  218.8730557501, 58.8451777196,  815.5005122179,
    276.5418342923, 231.0253535361, 176.8651683562, 447.5521775985,
    111.5091755825, 530.5373342108, 109.8497127001, 123.2327753974,
    310.6533568218, 295.1947544784, 390.4765697552, 352.1354289605,
    678.0538599638, 462.6107041793, 497.2383300955, 502.7988578218,
    677.1447412705, 151.4490411245, 302.7941329942, 170.8569591749,
    831.5889660036, 446.8142201578, 275.9830758086, 542.8406387836,
    275.5720122183, 458.3099125265, 685.8560826033, 389.0600963430,
    865.2117063625, 505.8001919745, 53.0697539456,  119.1974389274,
    414.3244671932, 172.9594672762, 833.5938371685, 514.5060210595,
    753.3553141131, 171.4892995390, 408.9293825100, 481.8294861833,
    150.5026648787, 374.0646366811, 59.5392883877,  188.3021796069,
    403.4492596246, 87.9648361362,  149.2817765509, 514.0240592069,
    400.3664425610, 282.7576273982, 490.9563636103, 864.4245125385,
    434.5025880835, 241.5939598859, 518.7482550566, 554.9144226227,
    418.9121090454, 176.3882196235, 501.6985827069, 717.8677221435,
    571.9711578413, 401.0524033215, 141.4036484201, 588.0335255417,
    518.6805706101, 46.1587447894,  185.9624450373, 628.5603237622,
    300.2353549779, 423.9310416875, 199.7668624213, 613.3138386735,
    168.5643779804, 770.0024877010, 134.4493610041, 506.0683819264,
    228.4669408101, 505.9756011107, 477.9580082725, 162.8023298529,
    578.0247069374, 220.8188696274, 780.6795433503, 487.0282920016,
    106.7505252062, 483.1833892297, 397.4235290323, 347.4835716058,
    379.0327006647, 335.7327639906, 435.2132943048, 460.3424282287,
    518.8354407434, 517.6422788506, 467.6563286575, 263.5152826410,
    67.3551502855,  262.1841622870, 272.6981295776, 557.0255003429,
    414.4081907078, 143.5780738203, 100.3176985943, 795.4839653411,
    258.2283411414, 725.9987875695, 171.3567065047, 742.2261042927,
    225.2902180486, 402.3605163850, 526.6176819152, 839.6252701217,
    86.0318001252,  644.5951362234, 312.5659219941, 216.9940605356,
    142.0552798749, 858.1362817072, 447.8740414717, 303.2645310663,
    190.5418717542, 583.8351784847, 460.6315474618, 661.0525331185,
    333.0271541029, 78.8238696870,  332.9181519373, 195.7248335329,
    26.0462914306,  49.9150358444,  348.7996921311, 367.1460722295,
    350.5750548702, 219.0303896976, 494.4970443295, 134.7993101833,
    454.0343325569, 459.7026291882, 610.4249553913, 275.9512817573,
    683.2942962369, 58.9218351202,  437.6072320145, 343.7579681065,
    217.2355881337, 187.6818455207, 780.2653050177, 54.1755620275,
    793.8638427958, 450.5591062561, 531.9581929737, 492.0213106321,
    473.0929801757, 59.3165422792,  725.6902487556, 255.0409546586,
    469.0802752041, 532.8418307729, 585.1618077951, 227.1127440639,
    445.6244598018, 263.7505388510, 542.6848313010, 314.5808597109,
    72.9027957178,  530.4753520951, 837.6080986158, 553.6094776227,
    133.8763190409, 198.2323133225, 492.4103065085, 428.9136537528,
    602.2810166767, 194.9065868559, 530.4821728677, 137.7881339182,
    671.3187715175, 340.0909063250, 274.9026436766, 158.5145919703,
    836.5795611316, 137.3222895207, 379.1497640784, 358.5904202584,
    679.0985494207, 191.1898320175, 440.1149969998, 298.3514698987,
    558.9174708628, 584.5875327235, 221.3779897711, 86.8817440628,
    639.4426229757, 543.2461101990, 731.4848363021, 331.1177709625,
    278.5014953865, 543.3677154586, 348.6873214038, 426.4291866668,
    205.5936429703, 118.0021599486, 780.1625854310, 336.1810324376,
    137.8852730576, 409.7864898255, 542.6187551700, 298.0789056286,
    427.5498459576, 433.5019084787, 72.6173691788,  399.3427105588,
    493.9939226899, 446.0814658246, 447.2346855519, 384.8084874401,
    121.3450288318, 716.9525378394, 379.5457463985, 303.8299315447,
    243.0927131019, 49.5854072703,  275.7770738745, 796.0438289275,
    512.8677904100, 771.7122317101, 187.3678727966, 783.9540477555,
    300.4906396052, 784.5413192732, 262.8273687195, 439.6668097272,
    224.8500098633, 570.6177534739, 180.7027376160, 593.1336304975,
    211.1059122177, 873.6102597045, 357.0731789842, 537.8849393372,
    324.9480864529, 599.5344512555, 378.4374648412, 610.2423383379,
    305.6506987114, 755.7390731148, 323.9257038562, 80.9324668293,
    550.1750907106, 25.3309136931,  329.0835899583, 797.2324071972,
    408.6889731146, 338.2281069443, 231.2765966915, 771.7815694667,
    424.9742779581, 775.9336105414, 168.0736784145, 565.7008876448,
    553.2851158248, 280.1042384662, 199.1224839809, 630.2602215701,
    287.3623955789, 853.7023318239, 317.1056761779, 636.0580374750,
    571.8755823217, 331.7892213117, 381.9852190492, 621.7007497512,
    302.2345596074, 306.2269843534, 66.5921896287,  464.4967898812,
    532.2438737668, 183.1994094074, 316.6446194318, 175.8102868542,
    32.1352499828,  175.6654098571, 636.4167862189, 384.1478000458,
    717.7256520875, 556.0747247524, 863.5540879634, 433.6879274570,
    695.3558134566, 408.7520546666, 464.7390727976, 510.7960732776,
    673.0642576082, 288.0066208795, 19.5000000000,  488.8484768999,
    246.3412718259, 176.0165460784, 630.4117789350, 377.5102343936,
    678.0506901675, 15.5000000000,  72.1881430350,  381.9918826176,
    522.3594221552, 76.3496554036,  719.0175853927, 421.6416859266,
    684.7743103355, 389.3870585218, 701.2154826542, 212.3269034922,
    45.4672139538,  230.6467364247, 392.1410648813, 211.3692877433,
    399.4318298072, 573.5114844734, 266.8353896072, 772.9894593883,
    174.5686341804, 629.3357392140, 522.8820859739, 471.8201133092,
    136.9431253078, 676.9585206233, 360.4750963941, 180.8082333832,
    133.6779694128, 464.7791760786, 115.4268068377, 683.0119870870,
    351.1414877226, 673.4231361268, 493.0454027643, 566.3273941574,
    543.2104582197, 694.5355922615, 397.0710473770, 466.0099153688,
    174.6305680378, 676.1875599859, 398.6634201945, 798.2353761535,
    242.2452906890, 34.2117485749,  460.6185991377, 553.7760466875,
    264.3035302257, 745.8959476089, 196.5336503989, 569.6717130333,
    313.5398134898, 653.4711706942, 312.3949365585, 553.1122941669,
    115.8350015418, 88.0574934860,  103.4593209967, 364.0813191909,
    490.2777855596, 669.1207734137, 340.4291641039, 215.5000000000,
    274.5000000000, 746.5026681196, 553.7963466366, 26.7085449647,
    303.5995250593, 408.2975324992, 417.2433095204, 105.3473696010,
    480.6906717391, 160.0593397081, 461.0607457886, 185.6792037312,
    32.2396839867,  448.9970814988, 856.0966303491, 475.6476857503,
    69.9660367550,  157.8302024485, 590.9213687486, 516.7408039169,
    337.2162556826, 385.1186778216, 695.6608332389, 19.4499167373,
    282.3832085524, 32.2450298994,  200.4395416672, 303.6404747500,
    70.5972773791,  264.8476323248, 474.9727702731, 190.0488457880,
    127.2391344988, 265.3736935060, 391.0818052628, 512.0873774802,
    281.7494029347, 238.3533851500, 789.4460645963, 421.8497393456,
    22.1855441364,  533.9091797419, 102.7870196793, 455.6423139238,
    820.5033574157, 521.6115331319, 120.9198875244, 67.7898642342,
    800.8973815545, 248.0620845540, 259.8932647738, 303.9826156374,
    43.8833694777,  174.3642853301, 827.2571618618, 325.1443791417,
    293.4852992538, 128.6428235168, 534.4013130503, 218.3014223875,
    370.2143285379, 467.8494000091, 148.4781111710, 514.1674244971,
    472.6572843767, 37.3922894762,  178.0935177833, 786.3026900309,
    463.5154387526, 368.3829433204, 457.7574769459, 510.3503920779,
    156.3121483915, 189.5198039577, 102.1391107593, 441.3128081982,
    420.4639684905, 746.5789998924, 208.5448051527, 680.3741890779,
    509.2523542884, 65.2866664593,  480.7189856753, 604.6095532971,
    278.5945589181, 864.1182822955, 508.5575766684, 785.4607473781,
    106.1162659222, 574.8807764325, 703.8781509075, 205.8461055811,
    542.5155223050, 269.7478936500, 534.1703443158, 397.6297207975,
    452.9409507873, 197.8211739759, 290.9268959730, 130.2649070452,
    271.4592984322, 115.7620372215, 508.7370017948, 354.3265658258,
    728.9520641391, 87.6634423925,  860.5706105563, 427.8758900894,
    419.9614509366, 89.9419220725,  93.7652622808,  388.9694433531,
    541.3453748904, 302.5857944090, 218.2587148128, 206.2042086365,
    805.5836531263, 158.9517994226, 864.5907589867, 260.5363881423,
    654.2560149092, 570.6199396471, 736.4839207001, 536.3317688372,
    779.9421124414, 302.8388899700, 660.4579529197, 484.8698019568,
    712.3589048614, 255.6707380298, 220.0546148816, 86.2740485189,
    674.2090670580, 428.7823455118, 595.1775407582, 534.8346698538,
    593.8955419385, 473.2078894912, 754.4565171143, 485.5991308067,
    669.5212716891, 497.2124802650, 508.9254389097, 80.6353488656,
    183.0943366161, 468.6811580547, 250.5363121991, 197.1434826458,
    214.4619622635, 555.4838359226, 389.7104658252, 147.8724064640,
    108.0244165238, 154.5041215787, 79.3847400707,  578.5000000000,
    260.0954987798, 119.1325263275, 798.9060071514, 152.3710273647,
    197.2946655195, 241.2763514495, 731.7269156878, 140.5079025632,
    410.3093326534, 571.2967969259, 167.6601606171, 60.9824058064,
    619.3476886445, 429.3357373956, 80.1028783289,  101.2553053161,
    616.8625687653, 516.8461972703, 710.0863554522, 501.7760521122,
    395.6627703700, 240.5758198466, 44.5154529353,  29.7220972992,
    612.2248605113, 164.9844215241, 776.2003527600, 106.5517711142,
    819.4281050454, 376.9814872720, 867.2210419883, 424.6259861651,
    531.5974366318, 381.3418709936, 463.5177109660, 518.9244576151,
    513.1795568652, 300.0025238808, 142.1159764892, 598.4406485025,
    513.8170038465, 519.3135261766, 491.8084854426, 674.7759008561,
    427.7987199186, 666.7386767578, 197.3058629777, 554.0481755838,
    359.9676814439, 497.4111502035, 650.1813219703, 500.9723127985,
    56.7890056053,  138.0389096226, 393.1157721859, 127.4267516156,
    111.2729713874, 105.5476989533, 232.8685725590, 260.0734857710,
    756.7445754986, 453.7315661727, 15.5000000000,  148.4549198025,
    115.2930740179, 42.3781986239,  229.7821548449, 86.8004186516,
    541.1132487513, 362.8120287381, 462.4939629415, 163.9362700178,
    183.0625951734, 241.3089983627, 156.0916461744, 299.6626012989,
    369.6107118368, 296.5680685652, 543.5218249451, 343.8130818413,
    875.8471521557, 365.0622574914, 50.6177152562,  382.2182579173,
    730.2532011224, 61.6786619302,  177.8494992886, 344.0165727931,
    646.5022870950, 429.2585430205, 240.4225504872, 509.3935871602,
    675.5861489945, 277.7655613210, 690.9251859464, 134.0803331939,
    150.0871678765, 361.2223364883, 243.8221538920, 105.1701142597,
    549.2739159634, 330.9405173159, 479.7383886459, 434.3402482106,
    770.2443285355, 467.9541838377, 724.7967923487, 446.1956607321,
    204.3244604265, 381.9975669919, 113.5594017090, 119.7454225899,
    25.0772489069,  466.9139323451, 627.1422566789, 424.3096410917,
    428.8829324688, 323.5587618743, 590.0320692829, 551.9451632706,
    618.6722696777, 533.0018739627, 119.8157006362, 542.7684242160,
    748.0763351995, 201.1109168381, 360.2361873150, 420.0204705020,
    110.6970754974, 401.3238092364, 150.9786226366, 546.0651999149,
    575.6487710569, 489.7393153702, 463.5902900009, 381.2264570629,
    381.3519498130, 468.9550175341, 65.8429731956,  513.3232940532,
    296.5213211026, 437.2442060434, 602.3995546970, 490.5037130965,
    762.9563299703, 134.4482185861, 541.2329701179, 421.0550673570,
    133.9958692155, 67.5731862276,  260.2322582214, 321.3165620647,
    32.5359518436,  390.7611331661, 67.7878643893,  335.6357316084,
    57.2038399880,  570.4850123525, 395.7596298285, 453.7440230126,
    298.3423528999, 97.5580347904,  694.1430164731, 226.3156096574,
    213.4850713901, 403.6321322793, 674.6308053819, 393.6867655443,
    450.6327651048, 535.4263045545, 142.7621194139, 538.9608085365,
    248.2871043044, 409.6079345149, 879.5000000000, 554.7375547988,
    448.3602507163, 544.6371450247, 468.4004631370, 530.1418366052,
    558.4232383210, 127.1736955139, 556.8700916592, 371.1191850872,
    350.7274604791, 539.1114140665, 123.5348812343, 558.1790593755,
    143.2078328499, 72.7237933446,  330.6975344240, 316.6235446090,
    328.2535489741, 340.9819356600, 429.7848416560, 276.3433330278,
    565.2458743246, 373.2939327828, 646.3155367202, 120.1339632853,
    767.2785329583, 341.5023597143, 376.5313624643, 420.0295265532,
    119.0352955174, 170.2550568440, 57.0202714641,  363.4738424333,
    735.8373374471, 209.7488790124, 262.0315724226, 61.0560075765,
    291.1736333538, 779.5938215932, 394.6647903298, 604.8498191345,
    218.3934755542, 15.5000000000,  238.4578676452, 705.0211274192,
    500.5341159618, 860.1755874539, 324.9194277072, 625.7899272373,
    144.7715791478, 381.4098951796, 353.0407163407, 183.9024647622,
    370.2693663855, 444.8860934928, 19.5000000000,  96.4429276975,
    398.9284313491, 671.4878663694, 581.9897862474, 419.3431211343,
    73.6716672886,  288.6345426315, 38.0989022262,  598.3286301691,
    68.6619082704,  333.4881127218, 150.2681618239, 616.4146794535,
    207.5400256830, 853.6829090152, 520.7111376439, 225.8491170605,
    474.8372490251, 216.7074524422, 439.9207118976, 871.5523254278,
    482.0818377889, 741.8900132312, 116.5788505778, 308.4623957870,
    324.2964853847, 518.6788697929, 386.0083426565, 464.8217571706,
    350.2567502019, 630.4676624524, 290.0632181526, 450.9299106771,
    465.6364070670, 500.4819880923, 566.0829786957, 807.8969160238,
    508.7832847058, 308.3340467926, 472.0807216480, 208.2522088705,
    160.3348812317, 271.4448074473, 69.8604365866,  570.1429122598,
    17.7932910504,  349.5132219245, 554.6141544949, 16.3753699184,
    143.4715562756, 748.1783990569, 350.3075431281, 725.8674600354,
    68.0541953008,  656.9876681725, 535.4170555010, 777.7861015686,
    568.1950006959, 292.7121747782, 270.9829747699, 291.7444687249,
    554.6621580750, 273.6367441197, 413.8266367177, 460.7526079064,
    99.2402415281,  298.8843950059, 717.9300151853, 333.7016662859,
    558.0190182033, 490.0813238413, 699.7345112405, 444.3735707052,
    549.4983827314, 169.5988928769, 115.2268875348, 542.8788183586,
    683.8933533579, 582.1892664432, 794.1042808091, 35.8553295109,
    812.3534504557, 310.8231029075, 176.9218246968, 73.5025929049,
    176.6711959107, 464.3164187485, 603.4666201797, 501.8954779097,
    148.9590191351, 391.9491798683, 486.0789261363, 388.1346759791,
    154.3906032807, 563.0551475333, 41.0161776026,  486.3997879309,
    240.3293604000, 29.5000000000,  587.7097513708, 331.4551787975,
    399.5133639019, 104.0939785438, 493.2747600594, 421.9757190320,
    376.3849210118, 497.4798687642, 867.1707574019, 345.2324905678,
    170.3254845631, 29.9862749355,  330.6913235816, 472.8136788569,
    402.2753562357, 822.2664722862, 556.6313996475, 596.1599229005,
    430.4066994492, 205.9707740383, 249.9608922555, 632.6439743160,
    234.7543688641, 798.6721001433, 471.9341437757, 745.3371430623,
    461.7654986283, 507.4056335594, 34.7341023261,  583.5105584534,
    478.0911288795, 105.8708225812, 113.2945618175, 465.4508595407,
    76.1354456035,  260.4710832581, 443.7009205743, 833.0821693411,
    292.4649297815, 447.3373085623, 82.0738116621,  627.8421788274,
    118.4762591417, 678.1343345532, 43.2483075677,  375.2241985385,
    381.8079300672, 582.9694839542, 174.1525323256, 866.9098579740,
    257.5338737504, 452.7134192941, 368.5827060843, 882.5000000000,
    182.2791872190, 672.3559204976, 435.4648218918, 744.1443766993,
    457.3179593166, 506.2703527984, 176.3208122804, 184.0787102660,
    256.2352099146, 722.1472752339, 192.1047547049, 694.3446908093,
    119.0161509320, 375.9585538793, 365.6496456167, 813.7726813953,
    411.9223138008, 503.3723515549, 360.9005498177, 852.1393750181,
    32.5000000000,  272.2517225641, 137.1954956714, 549.4801022975,
    326.2355741962, 249.6379857265, 599.5052733640, 473.4999527023,
    672.0437251685, 469.8457838967, 400.4405564972, 464.2546016065,
    768.0746958263, 230.7405981069, 709.2667591636, 241.2720723952,
    207.7743405267, 114.4490129585, 286.4096326438, 291.4627764319,
    747.6702041385, 582.4579433732, 543.1455371940, 200.9219963863,
    481.6300101363, 106.9856005277, 353.9937739625, 268.8836441253,
    572.1937805884, 116.9085139112, 216.5000000000, 266.5000000000,
    537.1949982742, 219.4512886424, 47.2106190185,  276.4239617768,
    649.1814880145, 172.3727345118, 281.9842787650, 441.3006750090,
    374.5762745067, 249.9389456452, 490.5823926519, 165.6396752409,
    859.9391568174, 328.1299759847, 507.8928061342, 235.9452700838,
    333.4729011926, 533.7907613021, 419.3509155797, 561.0126469936,
    61.6991671716,  476.0309479589, 413.2237918443, 469.4217608485,
    502.3156249516, 397.7587362444, 115.4950960109, 472.7524859332,
    589.6447793806, 315.4569410762, 220.7615822172, 326.8262785184,
    329.3257178164, 155.9035135277, 77.6787072028,  110.0956584715,
    161.3792718595, 426.4708600001, 845.7098527058, 511.2764121502,
    50.6250960631,  543.0741197817, 128.2047507023, 566.2798288414,
    345.2116531184, 302.4887409686, 485.4189121748, 245.4848603889,
    73.7158838558,  199.1618264821, 475.7640859392, 43.4889116967,
    142.5322709821, 221.0105162101, 767.1556720643, 75.3337802797,
    838.8156251957, 482.2085061759, 842.4450739249, 346.3385774260,
    570.1952457927, 428.0257649900, 109.0554286871, 306.8699653424,
    32.0398159778,  665.7692493874, 472.2959059029, 407.7800096974,
    224.1146084330, 684.6686923233, 483.1730281795, 214.1621186899,
    460.2779001753, 631.7300027209, 315.1900754254, 528.8657617724,
    466.9507227357, 17.2210983519,  181.0893944837, 188.6156741443,
    160.5097262899, 258.2838382355, 179.9887974339, 677.0419725309,
    217.3165725380, 708.9312197163, 238.3548651561, 130.2945546992,
    126.0970054917, 693.9045991032, 512.2156884277, 252.0933563879,
    119.8960288897, 64.0827193263,  89.0092556558,  624.5584140593,
    327.8781051636, 597.5486552104, 49.7224619985,  645.0500365476,
    451.4149822954, 535.2117252211, 440.7713227294, 690.3713855146,
    368.6896739299, 400.9769871139, 117.7526615978, 162.4247615392,
    462.6795451567, 491.0262810999, 476.4983924674, 626.9438819228,
    142.7699090896, 823.2390348783, 438.0843238300, 212.2648917517,
    188.6091087472, 645.3535190376, 88.4503918831,  64.1293184761,
    344.6456070430, 343.5234451691, 167.8290145273, 350.7692553422};

static void BM_DtoaShortest(benchmark::State& state) {
  char output[kFastDtoaMaximalLength + 10];
  Vector<char> buffer(output, sizeof(output));
  int length, decimal_point;
  unsigned idx = 0;
  for (auto _ : state) {
    bool ok = FastDtoa(kTestDoubles[idx++ % 4096], FAST_DTOA_SHORTEST, 0,
                       buffer, &length, &decimal_point);
    USE(ok);
  }
}

static void BM_DtoaSixDigits(benchmark::State& state) {
  char output[kFastDtoaMaximalLength + 10];
  Vector<char> buffer(output, sizeof(output));
  int length, decimal_point;
  unsigned idx = 0;
  for (auto _ : state) {
    bool ok = FastDtoa(kTestDoubles[idx++ % 4096], FAST_DTOA_PRECISION, 6,
                       buffer, &length, &decimal_point);
    USE(ok);
  }
}

BENCHMARK(BM_DtoaShortest);
BENCHMARK(BM_DtoaSixDigits);
