﻿/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using Lucene.Net.Search;
using Lucene.Net.Spatial.Prefix.Tree;
using Lucene.Net.Spatial.Queries;
using Spatial4n.Core.Shapes;

namespace Lucene.Net.Spatial.Prefix
{
	/// <summary>
	/// Based on {@link RecursivePrefixTreeFilter}.
	/// </summary>
	public class RecursivePrefixTreeStrategy : PrefixTreeStrategy
	{
		private int prefixGridScanLevel;

		public RecursivePrefixTreeStrategy(SpatialPrefixTree grid, string fieldName)
			: base(grid, fieldName)
		{
			prefixGridScanLevel = grid.GetMaxLevels() - 4;//TODO this default constant is dependent on the prefix grid size
		}

		public void SetPrefixGridScanLevel(int prefixGridScanLevel)
		{
			//TODO if negative then subtract from maxlevels
			this.prefixGridScanLevel = prefixGridScanLevel;
		}

		public override Filter MakeFilter(SpatialArgs args)
		{
			var op = args.Operation;
            if (op != SpatialOperation.Intersects)
				throw new UnsupportedSpatialOperation(op);

			Shape shape = args.Shape;

            int detailLevel = grid.GetLevelForDistance(args.ResolveDistErr(ctx, distErrPct));

			return new RecursivePrefixTreeFilter(GetFieldName(), grid, shape, prefixGridScanLevel, detailLevel);
		}

		public override string ToString()
		{
			return GetType().Name + "(prefixGridScanLevel:" + prefixGridScanLevel + ",SPG:(" + grid + "))";
		}
	}
}
